package com.almworks.jira.structure.api.effect;

import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.util.I18nText;
import com.atlassian.annotations.PublicApi;
import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.jira.bc.project.component.ProjectComponent;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.customfields.CustomFieldType;
import com.atlassian.jira.issue.customfields.impl.*;
import com.atlassian.jira.issue.customfields.option.Option;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.priority.Priority;
import com.atlassian.jira.issue.resolution.Resolution;
import com.atlassian.jira.issue.status.Status;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectConstant;
import com.atlassian.jira.project.version.Version;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.almworks.jira.structure.api.effect.StoredEffect.builder;
import static com.almworks.jira.structure.api.util.JiraFunc.*;

/**
 * Produces {@link StoredEffect effect descriptions} for common effects provided by Structure.
 */
@PublicApi
public final class CoreEffects {
  private CoreEffects() {
  }

  /**
   * Returns a description of an effect that can't be deserialized successfully but used to describe a specific problem.
   *
   * @param i18nText The warning message.
   * @param itemIdentities An ID list of items to which the warning message is related.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect emitWarning(@NotNull I18nText i18nText, @NotNull List<ItemIdentity> itemIdentities) {
    return builder("com.almworks.jira.structure:warning-effect-provider")
      .setParameter("i18nKey", i18nText.getI18nKey())
      .setParameter("arguments", Stream.of(i18nText.getArguments()).map(String::valueOf).collect(Collectors.toList()))
      .setParameter("affectedItems", itemIdentities.stream().map(String::valueOf).collect(Collectors.toList()))
      .build();
  }

  /**
   * Returns a description of an effect that would assign an issue to the given user. The reverse effect would
   * restore the assignee value that the issue had when the effect was applied.
   *
   * The returned effect would deserialize successfully only if the given user has the permission to the given issue,
   * and the assignee is assignable.
   *
   * @param issue The issue.
   * @param assignee The user to set. Passing {@code null} will make the issue unassigned.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect assignIssue(@NotNull Issue issue, @Nullable ApplicationUser assignee) {
    return builder("com.almworks.jira.structure:assignee-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("assignee", USER_KEY.la(assignee))
      .build();
  }

  /**
   * Returns a description of an effect that would set an issue reporter to the given user. The reverse effect
   * would restore the reporter that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param reporter The user to set. Passing {@code null} will clear the reporter value.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect setIssueReporter(@NotNull Issue issue, @Nullable ApplicationUser reporter) {
    return builder("com.almworks.jira.structure:reporter-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("reporter", USER_KEY.la(reporter))
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue priority to the given value. The reverse effect
   * would restore the priority that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param priority The priority to set. Passing {@code null} will clear the priority value.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setIssuePriority(@NotNull Issue issue, @Nullable Priority priority) {
    return builder("com.almworks.jira.structure:priority-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("priority", ISSUECONSTANT_ID.la(priority))
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue Description field to the given value.
   * The reverse effect would restore the Description field value that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param description The description value to set. Passing {@code null} will clear an issue description.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setIssueDescription(@NotNull Issue issue, @Nullable String description) {
    return builder("com.almworks.jira.structure:description-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("description", description)
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue Environment field to the given value.
   * The reverse effect would restore the environment that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param environment The environment value to set. Passing {@code null} will clear the Environment field value.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setIssueEnvironment(@NotNull Issue issue, @Nullable String environment) {
    return builder("com.almworks.jira.structure:environment-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("environment", environment)
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue Resolution field to the given value.
   * The reverse effect would restore the resolution that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param resolution The resolution to set. Passing {@code null} will clear the Resolution field value.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setIssueResolution(@NotNull Issue issue, @Nullable Resolution resolution) {
    return builder("com.almworks.jira.structure:resolution-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("resolution", ISSUECONSTANT_ID.la(resolution))
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue summary to the given value.
   * The reverse effect would restore the summary that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param summary The summary value to set. Passing {@code null} will clear the issue summary.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setIssueSummary(@NotNull Issue issue, @Nullable String summary) {
    return builder("com.almworks.jira.structure:summary-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("summary", summary)
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue due date to the given value.
   * The reverse effect would restore the due date value that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param date The due date to set. Passing {@code null} will clear the issue due date.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setIssueDueDate(@NotNull Issue issue, @Nullable Date date) {
    return builder("com.almworks.jira.structure:due-date-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("dueDate", date == null ? null : date.getTime())
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue resolution date to the given value.
   *
   * @param issue The issue.
   * @param date The resolution date to set. Passing {@code null} will clear the issue resolution date.
   * @return the effect description.
   *
   * @deprecated The "Resolved" field in Jira is not editable, and this method
   * should not be used. There is no corresponding effect provider in Structure.
   */
  @NotNull
  @Deprecated
  public static StoredEffect setIssueResolutionDate(@NotNull Issue issue, @Nullable Date date) {
    return builder("com.almworks.jira.structure:resolution-date-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("resolutionDate", date == null ? null : date.getTime())
      .build();
  }

  /**
   * Returns a description of an effect that would transition an issue to the given
   * status. The reverse effect would transition the issue to the status it had
   * when the effect was applied.
   *
   * The returned effect would deserialize successfully only if there's exactly
   * one direct transition from the issue's current status to the given status,
   * and that transition has no associated screen.
   *
   * @param issue The issue.
   * @param status The status. Passing {@code null} is allowed, but would produce
   *        an effect that would always deserialize with an error.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect setIssueStatus(@NotNull Issue issue, @Nullable Status status) {
    return builder("com.almworks.jira.structure:status-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("status", ISSUECONSTANT_ID.la(status))
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue original estimate to the given value.
   * The reverse effect would restore the original estimate value that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param originalEstimate The original estimate to set. Passing {@code null} will clear the issue original estimate.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setOriginalEstimate(@NotNull Issue issue, @Nullable Long originalEstimate) {
    return builder("com.almworks.jira.structure:original-estimate-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("originalEstimate", originalEstimate)
      .build();
  }

  /**
   * Returns a description of an effect that would change an issue remaining estimate to the given value.
   * The reverse effect would restore the remaining estimate value that the issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param remainingEstimate The remaining estimate to set. Passing {@code null} will clear the issue remaining estimate.
   * @return the effect description.
   */
  @NotNull
  public static StoredEffect setRemainingEstimate(@NotNull Issue issue, @Nullable Long remainingEstimate) {
    return builder("com.almworks.jira.structure:remaining-estimate-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("remainingEstimate", remainingEstimate)
      .build();
  }

  /**
   * Returns a description of an effect that would set a single select custom
   * field to the given option. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a single select custom field.
   * @param value The option to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a single select field.
   */
  @NotNull
  public static StoredEffect setSingleSelectCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable Option value) {
    checkCustomFieldType(customField, SelectCFType.class);
    return builder("com.almworks.jira.structure:single-select-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", OPTION_ID.la(value))
      .build();
  }

  /**
   * Returns a description of an effect that would set a project picker custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a project picker custom field.
   * @param value The project value to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a project picker field.
   */
  @NotNull
  public static StoredEffect setProjectPickerCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable Project value) {
    checkCustomFieldType(customField, ProjectCFType.class);
    return builder("com.almworks.jira.structure:project-picker-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", PROJECT_ID.la(value))
      .build();
  }

  /**
   * Returns a description of an effect that would set a single user custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a single user custom field.
   * @param value The user to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a single user field.
   */
  @NotNull
  public static StoredEffect setSingleUserCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable ApplicationUser value) {
    checkCustomFieldType(customField, UserCFType.class);
    return builder("com.almworks.jira.structure:single-user-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", USER_KEY.la(value))
      .build();
  }

  /**
   * Returns a description of an effect that would set a single version custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * The returned effect would deserialize successfully only if the given version is not archived and belongs to the
   * same project as the given issue.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a single version custom field.
   * @param value The version to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a single version field.
   */
  @NotNull
  public static StoredEffect setSingleVersionCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable Version value) {
    checkCustomFieldTypeIsMultiple(customField, VersionCFType.class, VersionCFType::isMultiple, false);
    return builder("com.almworks.jira.structure:single-version-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", PROJECTCONSTANT_ID.la(value))
      .build();
  }

  /**
   * Returns a description of an effect that would set a text custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a text custom field.
   * @param value The text value to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a text field.
   */
  @NotNull
  public static StoredEffect setTextCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable String value) {
    checkCustomFieldType(customField, GenericTextCFType.class);
    return builder("com.almworks.jira.structure:text-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", value)
      .build();
  }

  /**
   * Returns a description of an effect that would set a number custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a number custom field.
   * @param value The number value to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a number field.
   */
  @NotNull
  public static StoredEffect setNumberCustomField(@NotNull Issue issue, @NotNull CustomField customField,
    @Nullable Number value)
  {
    checkCustomFieldType(customField, NumberCFType.class);
    if (value instanceof BigInteger) value = value.longValue();
    if (value instanceof BigDecimal) value = value.doubleValue();
    return builder("com.almworks.jira.structure:number-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", value)
      .build();
  }

  /**
   * Returns a description of an effect that would set a date custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a date custom field.
   * @param value The date value to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a date field.
   */
  @NotNull
  public static StoredEffect setDateCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable Date value) {
    checkCustomFieldType(customField, DateCFType.class);
    return builder("com.almworks.jira.structure:date-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", value == null ? null : value.getTime())
      .build();
  }

  /**
   * Returns a description of an effect that would set a date time custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a date time custom field.
   * @param value The date time value to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a date time field.
   */
  @NotNull
  public static StoredEffect setDateTimeCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable Date value) {
    checkCustomFieldType(customField, DateTimeCFType.class);
    return builder("com.almworks.jira.structure:date-time-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", value == null ? null : value.getTime())
      .build();
  }

  /**
   * Returns a description of an effect that would set a single group custom
   * field to the given value. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a single group custom field.
   * @param value The group value to set. Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a single group field.
   */
  @NotNull
  public static StoredEffect setSingleGroupCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable Group value) {
    checkCustomFieldTypeIsMultiple(customField, MultiGroupCFType.class, MultiGroupCFType::isMultiple, false);
    return builder("com.almworks.jira.structure:single-group-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", value == null ? null : value.getName())
      .build();
  }

  /**
   * Returns a description of an effect that would set a cascading select custom
   * field to the given option. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue The issue.
   * @param customField The custom field, must be a cascading select custom field.
   * @param value The option to set. Can be a parent or a child option.
   *        Passing {@code null} will clear the field value.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a cascading select field.
   */
  @NotNull
  public static StoredEffect setCascadeCustomField(@NotNull Issue issue, @NotNull CustomField customField, @Nullable Option value) {
    checkCustomFieldType(customField, CascadingSelectCFType.class);
    return builder("com.almworks.jira.structure:cascade-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter("customField", customField.getId())
      .setParameter("value", OPTION_ID.la(value))
      .build();
  }

  /**
   * Returns a description of an effect that would replace issue labels with the given values.
   * The inverse effect would restore the value that the issue labels had when the effect was applied.
   *
   * @param issue  The issue.
   * @param labels The labels to set.
   * @return The effect description.
   */
  public static StoredEffect setIssueLabels(Issue issue, Collection<String> labels) {
    return updateLabels(issue, labels, "set");
  }

  /**
   * Returns a description of an effect that would add labels to the issue.
   * The inverse effect would remove the added labels from the issue.
   *
   * @param issue  The issue.
   * @param labels The labels to add.
   * @return The effect description.
   */
  public static StoredEffect addIssueLabels(Issue issue, Collection<String> labels) {
    return updateLabels(issue, labels, "add");
  }

  /**
   * Returns a description of an effect that would remove labels from the issue.
   * The inverse effect would add the removed labels back to the issue.
   *
   * @param issue  The issue.
   * @param labels The labels to remove.
   * @return The effect description.
   */
  public static StoredEffect removeIssueLabels(Issue issue, Collection<String> labels) {
    return updateLabels(issue, labels, "remove");
  }

  /**
   * Returns a description of an effect that would replace issue components with the given values.
   * The inverse effect would restore the value that the issue components had when the effect was applied.
   *
   * @param issue      The issue.
   * @param components The components to set.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect setIssueComponents(@NotNull Issue issue, @NotNull Collection<ProjectComponent> components) {
    return updateComponents(issue, components, "set");
  }

  /**
   * Returns a description of an effect that would add components to the issue.
   * The inverse effect would remove the added components from the issue.
   *
   * @param issue      The issue.
   * @param components The components to add.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect addIssueComponents(@NotNull Issue issue, @NotNull Collection<ProjectComponent> components) {
    return updateComponents(issue, components, "add");
  }

  /**
   * Returns a description of an effect that would remove components from the issue.
   * The inverse effect would add the removed components back to the issue.
   *
   * @param issue      The issue.
   * @param components The components to remove.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect removeIssueComponents(@NotNull Issue issue, @NotNull Collection<ProjectComponent> components) {
    return updateComponents(issue, components, "remove");
  }

  /**
   * Returns a description of an effect that would replace issue affected versions with the given values.
   * The inverse effect would restore the value that the issue affected versions had when the effect was applied.
   *
   * @param issue    The issue.
   * @param versions The versions to set.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect setIssueAffectedVersions(@NotNull Issue issue, @NotNull Collection<Version> versions) {
    return updateAffectedVersions(issue, versions, "set");
  }

  /**
   * Returns a description of an effect that would add affected versions to the issue.
   * The inverse effect would remove the added affected versions from the issue.
   *
   * @param issue    The issue.
   * @param versions The versions to add.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect addIssueAffectedVersions(@NotNull Issue issue, @NotNull Collection<Version> versions) {
    return updateAffectedVersions(issue, versions, "add");
  }

  /**
   * Returns a description of an effect that would remove affected versions from the issue.
   * The inverse effect would add the removed affected versions back to the issue.
   *
   * @param issue    The issue.
   * @param versions The versions to remove.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect removeIssueAffectedVersions(@NotNull Issue issue, @NotNull Collection<Version> versions) {
    return updateAffectedVersions(issue, versions, "remove");
  }

  /**
   * Returns a description of an effect that would replace issue fix versions with the given values.
   * The inverse effect would restore the value that the issue fix versions had when the effect was applied.
   *
   * @param issue    The issue.
   * @param versions The versions to set.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect setIssueFixVersions(@NotNull Issue issue, @NotNull Collection<Version> versions) {
    return updateFixVersions(issue, versions, "set");
  }

  /**
   * Returns a description of an effect that would add fix versions to the issue.
   * The inverse effect would remove the added fix versions from the issue.
   *
   * @param issue    The issue.
   * @param versions The versions to add.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect addIssueFixVersions(@NotNull Issue issue, @NotNull Collection<Version> versions) {
    return updateFixVersions(issue, versions, "add");
  }

  /**
   * Returns a description of an effect that would remove fix versions from the issue.
   * The inverse effect would add the removed fix versions back to the issue.
   *
   * @param issue    The issue.
   * @param versions The versions to remove.
   * @return The effect description.
   */
  @NotNull
  public static StoredEffect removeIssueFixVersions(@NotNull Issue issue, @NotNull Collection<Version> versions) {
    return updateFixVersions(issue, versions, "remove");
  }

  /**
   * Returns a description of an effect that would set a multi version custom
   * field to the given versions. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi version custom field.
   * @param versions    The versions to set.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi version field.
   */
  @NotNull
  public static StoredEffect setCustomFieldVersions(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Version> versions)
  {
    return updateMultiVersionCustomField(issue, customField, versions, "set");
  }

  /**
   * Returns a description of an effect that would add the given versions to a multi version custom field.
   * The inverse effect would remove the added versions from the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi version custom field.
   * @param versions    The versions to add.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi version field.
   */
  @NotNull
  public static StoredEffect addCustomFieldVersions(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Version> versions)
  {
    return updateMultiVersionCustomField(issue, customField, versions, "add");
  }

  /**
   * Returns a description of an effect that would remove the given versions from a multi version custom field.
   * The inverse effect would add the removed versions back to the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi version custom field.
   * @param versions    The versions to remove.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi version field.
   */
  @NotNull
  public static StoredEffect removeCustomFieldVersions(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Version> versions)
  {
    return updateMultiVersionCustomField(issue, customField, versions, "remove");
  }

  /**
   * Returns a description of an effect that would set a multi select custom
   * field to the given options. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi select custom field.
   * @param options     The options to set.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi select field.
   */
  @NotNull
  public static StoredEffect setCustomFieldOptions(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Option> options)
  {
    return updateMultiSelectCustomField(issue, customField, options, "set");
  }

  /**
   * Returns a description of an effect that would add the given options to a multi select custom field.
   * The inverse effect would remove the added options from the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi select custom field.
   * @param options     The options to add.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi select field.
   */
  @NotNull
  public static StoredEffect addCustomFieldOptions(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Option> options)
  {
    return updateMultiSelectCustomField(issue, customField, options, "add");
  }

  /**
   * Returns a description of an effect that would remove the given options from a multi select custom field.
   * The inverse effect would add the removed options back to the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi select custom field.
   * @param options     The options to remove.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi select field.
   */
  @NotNull
  public static StoredEffect removeCustomFieldOptions(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Option> options)
  {
    return updateMultiSelectCustomField(issue, customField, options, "remove");
  }

  /**
   * Returns a description of an effect that would set a multi user custom
   * field to the given users. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi user custom field.
   * @param users       The users to set.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi user field.
   */
  @NotNull
  public static StoredEffect setCustomFieldUsers(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<ApplicationUser> users)
  {
    return updateMultiUserCustomField(issue, customField, users, "set");
  }

  /**
   * Returns a description of an effect that would add the given users to a multi user custom field.
   * The inverse effect would remove the added users from the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi user custom field.
   * @param users       The users to add.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi user field.
   */
  @NotNull
  public static StoredEffect addCustomFieldUsers(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<ApplicationUser> users)
  {
    return updateMultiUserCustomField(issue, customField, users, "add");
  }

  /**
   * Returns a description of an effect that would remove the given users from a multi user custom field.
   * The inverse effect would add the removed users back to the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi user custom field.
   * @param users       The users to remove.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi user field.
   */
  @NotNull
  public static StoredEffect removeCustomFieldUsers(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<ApplicationUser> users)
  {
    return updateMultiUserCustomField(issue, customField, users, "remove");
  }

  /**
   * Returns a description of an effect that would set a multi group custom
   * field to the given groups. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi group custom field.
   * @param groups      The groups to set.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi group field.
   */
  @NotNull
  public static StoredEffect setCustomFieldGroups(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Group> groups)
  {
    return updateMultiGroupCustomField(issue, customField, groups, "set");
  }

  /**
   * Returns a description of an effect that would add the given groups to a multi group custom field.
   * The inverse effect would remove the added groups from the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi group custom field.
   * @param groups      The groups to add.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi group field.
   */
  @NotNull
  public static StoredEffect addCustomFieldGroups(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Group> groups)
  {
    return updateMultiGroupCustomField(issue, customField, groups, "add");
  }

  /**
   * Returns a description of an effect that would remove the given groups from a multi group custom field.
   * The inverse effect would add the removed groups back to the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a multi group custom field.
   * @param groups      The groups to remove.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a multi group field.
   */
  @NotNull
  public static StoredEffect removeCustomFieldGroups(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Group> groups)
  {
    return updateMultiGroupCustomField(issue, customField, groups, "remove");
  }

  /**
   * Returns a description of an effect that would set a labels custom
   * field to the given labels. The inverse effect would restore the value that
   * the given issue had when the effect was applied.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a labels custom field.
   * @param labels      The labels to set.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a labels field.
   */
  @NotNull
  public static StoredEffect setCustomFieldLabels(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<String> labels)
  {
    return updateLabelsCustomField(issue, customField, labels, "set");
  }

  /**
   * Returns a description of an effect that would add the given labels to a labels custom field.
   * The inverse effect would remove the added labels from the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a labels custom field.
   * @param labels      The labels to add.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a labels field.
   */
  @NotNull
  public static StoredEffect addCustomFieldLabels(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<String> labels)
  {
    return updateLabelsCustomField(issue, customField, labels, "add");
  }

  /**
   * Returns a description of an effect that would remove the given labels from a labels custom field.
   * The inverse effect would add the removed labels back to the custom field.
   *
   * @param issue       The issue.
   * @param customField The custom field, must be a labels custom field.
   * @param labels      The labels to remove.
   * @return The effect description.
   * @throws IllegalArgumentException if the given custom field is not a labels field.
   */
  @NotNull
  public static StoredEffect removeCustomFieldLabels(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<String> labels)
  {
    return updateLabelsCustomField(issue, customField, labels, "remove");
  }

  private static StoredEffect updateLabels(@NotNull Issue issue, @NotNull Collection<String> labels, @NotNull String operation) {
    return builder("com.almworks.jira.structure:labels-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(labels, String::trim))
      .build();
  }

  private static StoredEffect updateComponents(@NotNull Issue issue, @NotNull Collection<ProjectComponent> components, @NotNull String operation) {
    return builder("com.almworks.jira.structure:components-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(components, ProjectConstant::getId))
      .build();
  }

  private static StoredEffect updateAffectedVersions(@NotNull Issue issue, @NotNull Collection<Version> versions, @NotNull String operation) {
    return builder("com.almworks.jira.structure:affects-versions-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(versions, ProjectConstant::getId))
      .build();
  }

  private static StoredEffect updateFixVersions(@NotNull Issue issue, @NotNull Collection<Version> versions, @NotNull String operation) {
    return builder("com.almworks.jira.structure:fix-versions-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(versions, ProjectConstant::getId))
      .build();
  }

  private static StoredEffect updateMultiVersionCustomField(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Version> versions, @NotNull String operation)
  {
    checkCustomFieldTypeIsMultiple(customField, VersionCFType.class, VersionCFType::isMultiple, true);
    return builder("com.almworks.jira.structure:multi-versions-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(versions, ProjectConstant::getId))
      .setParameter("customField", customField.getId())
      .build();
  }

  private static StoredEffect updateMultiSelectCustomField(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Option> options, @NotNull String operation)
  {
    checkCustomFieldType(customField, MultiSelectCFType.class);
    return builder("com.almworks.jira.structure:multi-select-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(options, Option::getOptionId))
      .setParameter("customField", customField.getId())
      .build();
  }

  private static StoredEffect updateMultiUserCustomField(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<ApplicationUser> users, @NotNull String operation)
  {
    checkCustomFieldType(customField, MultiUserCFType.class);
    return builder("com.almworks.jira.structure:multi-user-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(users, ApplicationUser::getKey))
      .setParameter("customField", customField.getId())
      .build();
  }

  private static StoredEffect updateMultiGroupCustomField(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<Group> groups, @NotNull String operation)
  {
    checkCustomFieldTypeIsMultiple(customField, MultiGroupCFType.class, MultiGroupCFType::isMultiple, true);
    return builder("com.almworks.jira.structure:multi-group-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(groups, Group::getName))
      .setParameter("customField", customField.getId())
      .build();
  }

  private static StoredEffect updateLabelsCustomField(@NotNull Issue issue, @NotNull CustomField customField,
    @NotNull Collection<String> labels, @NotNull String operation)
  {
    checkCustomFieldType(customField, LabelsCFType.class);
    return builder("com.almworks.jira.structure:multi-labels-custom-field-effect-provider")
      .setParameter("issue", issue.getId())
      .setParameter(operation, unique(labels, String::trim))
      .setParameter("customField", customField.getId())
      .build();
  }

  @NotNull
  private static <T> List<Object> unique(@NotNull Collection<T> values, @NotNull Function<? super T, Object> serializer) {
    return values.stream().map(serializer).distinct().collect(Collectors.toList());
  }

  private static void checkCustomFieldType(
    @NotNull CustomField customField, @NotNull Class<? extends CustomFieldType> expectedType)
    throws IllegalArgumentException
  {
    CustomFieldType actualType = customField.getCustomFieldType();
    if (!expectedType.isInstance(actualType)) {
      throw new IllegalArgumentException(String.format(
        "Expected a custom field of type %s, but %s \"%s\" is of type %s",
        expectedType, customField.getId(), customField.getName(), actualType));
    }
  }

  private static <T extends CustomFieldType> void checkCustomFieldTypeIsMultiple(@NotNull CustomField customField, @NotNull Class<T> customFieldType,
    @NotNull Function<T, Boolean> getter, boolean mustBeMultiple)
  {
    checkCustomFieldType(customField, customFieldType);
    @SuppressWarnings("unchecked")
    T cfType = (T) customField.getCustomFieldType();
    if (getter.apply(cfType) != mustBeMultiple) {
      if (mustBeMultiple) {
        throw new IllegalArgumentException("Custom field must be multiple, but it's not: " + customField.getId());
      } else {
        throw new IllegalArgumentException("Custom field must not be multiple, but it is: " + customField.getId());
      }
    }
  }
}
