package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.ValueFormat;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.EnumSet;
import java.util.Set;

/**
 * <p>Allows the attribute loader to declare that the value it produces is calculated using some of the values from the context.</p>
 *
 * <p>Different context dependencies may have different effects. Some, like {@link #USER_LOCALE}, provide "compartments" for values,
 * allowing to efficiently cache values for each locale. Some, like {@link #CURRENT_TIME}, declare the dependency on time and the corresponding
 * caching rules will be applied.</p>
 *
 * <p>Context dependencies propagate through dependencies. Making an attribute dependent on the current user will also make all dependent attribute
 * dependent on the current user.</p>
 *
 * @see AttributeLoader#getContextDependencies()
 */
@PublicApi
public enum AttributeContextDependency {
  /**
   * <p>The values depend on the current user.</p>
   *
   * <p>The values will be calculated separately for each individual user (including anonymous user).</p>
   *
   * <p>Use {@link AttributeLoaderContext#getUser()} when calculating the value to get the current user.</p>
   *
   * <p>This context dependency supersedes {@link #USER_LOCALE} and {@link #USER_TIMEZONE}. No need to declare these context dependencies, since
   * the user compartment already provides the necessary separation for different locales and time zones.</p>
   *
   * <p>All values with this context dependency will be invalidated when the user itself is changed.</p>
   */
  USER,

  /**
   * <p>The values depend on the current user's locale. Typically this is used when calculating a HTML-rendered value.</p>
   *
   * <p>The values will be calculated separately for each locale. If two users have the same locale, they will share the same value.</p>
   *
   * <p>Use {@link AttributeLoaderContext#getLocale()} or {@link AttributeLoaderContext#getI18nHelper()} when calculating the value.</p>
   */
  USER_LOCALE,

  /**
   * <p>The values depend on the current user's time zone.</p>
   *
   * <p>The values will be calculated separately for each time zone. If two users have the same time zone, they will share the same value.</p>
   *
   * <p>Use {@link AttributeLoaderContext#getTimeZone()} when calculating the value.</p>
   */
  USER_TIMEZONE,

  /**
   * <p>The values depend on the structure that they are being shown in.</p>
   *
   * <p>The values may depend only on the structure ID, without any transformations. When calculating the values for a query-based forest,
   * the structure ID will be equal to {@code 0}, and so all the values calculated for queries will be shared.</p>
   *
   * <p>Note that forest-dependent attributes will naturally be stored in a forest-specific cache. There's no need to declare
   * structure dependency for aggregates, for example. You only need to declare the dependency if you actively use the structure ID
   * for calculating the value. For example, you can store different values for the same items in different structures.</p>
   *
   * <p>Use {@link AttributeLoaderContext#getBaseStructureId()} when calculating the value.</p>
   */
  STRUCTURE,

  /**
   * <p>The values depend on the current time.</p>
   *
   * <p>Declaring this context dependency will cause the values to expire after some time and get recalculated. You can define the expiration
   * time by calling {@link AttributeLoaderContext#valueExpires}, or just declare the context dependency and go with the default expiration.</p>
   *
   * <p>You can use {@link AttributeLoaderContext#getLoadTimeMillis()} and {@link AttributeLoaderContext#getLoadTimeNanos()} when calculating
   * the value. These methods will return a consistent number for all values loaded in the same loading process. But you can also use
   * {@link System#currentTimeMillis()} and {@link System#nanoTime()}.</p>
   */
  CURRENT_TIME,

  /**
   * <p>The values depend on some items, which will be added via {@link AttributeLoaderContext#addTrail}.</p>
   *
   * <p>You need to declare this dependency in order for the system to be prepared for the {@code addTrail()} call. In particular, multi-row
   * attributes that declare this dependency will not be eligible for pre-validation (reusing a previously calculated value if nothing has
   * changed in the inputs).</p>
   *
   * <p>Note that you <em>don't need</em> to declare this dependency if you're declaring a global trail via {@link AttributeLoader#getGlobalTrail()}.</p>
   */
  TRAIL;

  public static Set<AttributeContextDependency> guessContextDependencies(AttributeSpec<?> spec) {
    if (ValueFormat.HTML.equals(spec.getFormat())) {
      return EnumSet.of(USER);
    } else {
      return null;
    }
  }

  @NotNull
  public static Set<AttributeContextDependency> union(@Nullable Set<AttributeContextDependency> set, @Nullable AttributeContextDependency... additional) {
    Set<AttributeContextDependency> result = EnumSet.noneOf(AttributeContextDependency.class);
    if (set != null) {
      result.addAll(set);
    }
    if (additional != null) {
      for (AttributeContextDependency dependency : additional) {
        if (dependency != null) {
          result.add(dependency);
        }
      }
    }
    return result;
  }
}
