package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.util.*;
import com.atlassian.annotations.PublicApi;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Map;

/**
 * <p>{@code ValueFormat} is used as a part of {@link AttributeSpec} to define in what format the value should be
 * returned.</p>
 *
 * <p>There are a number of well-known formats, defined in this class. However, custom formats can be
 * created and used on the server side by Structure extensions. Only the standard formats can be sent over via REST API.</p>
 *
 * <p>It is important to note the difference between value format and value type. The format defines value type but
 * also the semantics of the value. For example, both {@link #HTML} and {@link #TEXT} formats provide {@code String}
 * values, but HTML values can actually be rendered as-is on a web page (without HTML escaping) and they are unsuitable
 * to be shown to the user as plain text.</p>
 *
 * <p>Different value types must have different IDs. When creating a new value type, it's recommended to have your
 * package prefix as a part of the value type ID.</p>
 *
 * @param <T> value type
 */
@PublicApi
public final class ValueFormat<T> {
  /**
   * HTML values can be shown on a web page. Most attributes can provide HTML values.
   */
  public static final ValueFormat<String> HTML = new ValueFormat<>("html", String.class);

  /**
   * TEXT values are plain text. If shown on a web page, they must be HTML-escaped.
   */
  public static final ValueFormat<String> TEXT = new ValueFormat<>("text", String.class);

  /**
   * ID is a special format for values that represent entities. Some attributes may provide ID format in case
   * the client needs only the ID of the value. If the ID is numeric, it is converted to String.
   */
  public static final ValueFormat<String> ID = new ValueFormat<>("id", String.class);

  /**
   * NUMBER values are numeric and usually are either {@code Long} or {@code Double}. When interpreting NUMBER values,
   * try to avoid casts and use {@link Number#longValue()} and {@link Number#doubleValue()} instead.
   */
  public static final ValueFormat<Number> NUMBER = new ValueFormat<>("number", Number.class);

  /**
   * BOOLEAN value contains simple boolean value.
   */
  public static final ValueFormat<Boolean> BOOLEAN = new ValueFormat<>("boolean", Boolean.class);

  /**
   * TIME values contain Epoch time in milliseconds.
   */
  public static final ValueFormat<Long> TIME = new ValueFormat<>("time", Long.class);

  /**
   * DURATION format contain the number of milliseconds between two points in time.
   */
  public static final ValueFormat<Long> DURATION = new ValueFormat<>("duration", Long.class);

  /**
   * JSON_OBJECT values are Java maps, ready to be converted to a JSON object. The maps must contain only
   * String keys and only JSON-compatible values: numbers, strings, booleans, maps or lists.
   */
  public static final ValueFormat<Map> JSON_OBJECT = new ValueFormat<>("json", Map.class);

  /**
   * JSON_ARRAY values are Java lists, ready to be converted to a JSON array. The lists must contain only
   * JSON-compatible values: numbers, strings, booleans, maps or lists.
   */
  public static final ValueFormat<List> JSON_ARRAY = new ValueFormat<>("jsonArray", List.class);

  /**
   * ANY format is suitable when the requesting code can accept object of any type and there's no knowledge
   * at development time what format of the attribute is going to be available. This format is used for nested
   * attribute specs that can provide results of any format. The Java type may be any class.
   */
  public static final ValueFormat<Object> ANY = new ValueFormat<>("any", Object.class);

  /**
   * <p>ORDER values are special values that can be used to sort by this attributes. The Java type of the value may
   * be any class, as long as a) they are {@link Comparable}, and b) the ORDER values for the same item type will be the
   * same Java type.</p>
   *
   * <p>Values are compared using {@link TotalOrder} or {@link ComparableTuple}.</p>
   */
  public static final ValueFormat<Comparable> ORDER = new ValueFormat<>("order", Comparable.class);

  private static final ValueFormat<?>[] STANDARD_FORMATS = {
    HTML, TEXT, NUMBER, BOOLEAN, DURATION, TIME, JSON_OBJECT, JSON_ARRAY, ORDER, ID, ANY
  };

  @NotNull
  private final String myFormatId;

  @NotNull
  private final Class<T> myValueClass;

  /**
   * Creates a new value format.
   *
   * @param formatId unique format ID (use your package prefix to guarantee uniqueness). Must not contain colon (":")
   * or be unreasonably long.
   * @param valueClass Java type of the values
   */
  public ValueFormat(@NotNull String formatId, @NotNull Class<T> valueClass) {
    if (StringUtils.isBlank(formatId)) {
      throw new IllegalArgumentException("format id must not be empty");
    }
    if (formatId.length() > Limits.MAX_SPEC_ID_LENGTH) {
      throw new IllegalArgumentException("format id must not be longer than " + Limits.MAX_SPEC_ID_LENGTH + " chars");
    }
    if (formatId.indexOf(':') >= 0) {
      throw new IllegalArgumentException("format id must not contain colon");
    }
    //noinspection ConstantConditions
    if (valueClass == null) {
      throw new IllegalArgumentException("format class must be not null");
    }
    myFormatId = formatId;
    myValueClass = valueClass;
  }

  /**
   * Returns the unique ID of the value format.
   */
  @NotNull
  public String getFormatId() {
    return myFormatId;
  }

  /**
   * Returns the Java class of the values in this format.
   */
  @NotNull
  public Class<T> getValueClass() {
    return myValueClass;
  }

  /**
   * <p>Performs type checking and cast of an arbitrary attribute specification to the given format.
   * This method should be used only to cast {@code AttributeSpec} type parameter to the desired format.
   * It does not convert {@code AttributeSpec}, so if the format is different, an exception will be thrown.</p>
   *
   * @param spec attribute spec with unknown type
   * @return attribute spec with type {@code T}
   * @throws ClassCastException in case attribute specification is of different format
   * @see AttributeSpec#as(ValueFormat)
   */
  @NotNull
  public AttributeSpec<T> cast(@NotNull AttributeSpec<?> spec) {
    if (this.equals(spec.getFormat())) {
      //noinspection unchecked
      return (AttributeSpec<T>) spec;
    } else {
      throw new ClassCastException("cannot cast attribute " + spec + " as format " + this);
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    return myFormatId.equals(((ValueFormat<?>) o).myFormatId);
  }

  @Override
  public int hashCode() {
    return myFormatId.hashCode();
  }

  public String toString() {
    return myFormatId;
  }

  /**
   * Returns a standard format (declared in this class) given its format ID.
   *
   * @param formatId format ID
   * @return the standard format or {@code null} if there's no such standard format
   */
  @Nullable
  public static ValueFormat<?> getStandardFormat(String formatId) {
    if (formatId == null) return null;
    for (ValueFormat<?> format : STANDARD_FORMATS) {
      if (format.getFormatId().equals(formatId)) {
        return format;
      }
    }
    return null;
  }
}
