package com.almworks.jira.structure.api.item;

import com.almworks.jira.structure.api.error.StructureRuntimeException;

import javax.annotation.Nonnull;

/**
 * <p>{@code ItemTypeRegistry} is an internal service that provides mapping of all known item type keys (module keys)
 * to int values (primary keys in a db table).</p>
 *
 * <p>The underlying table is append-only; once the key is encountered, it is enumerated and will be kept in the
 * database forever.</p>
 *
 * <p>The values of item type registry must not be exported during backup, or during any other export process. They
 * are local to this instance.</p>
 */
public interface ItemTypeRegistry {
  /**
   * Retrieves the key of an item type, based on its ID. The ID must be previously received by calling {@link #getOrCreateTypeId(String)}.
   *
   * @param id the ID of the item type
   * @return the key of the item type, never null
   * @throws ItemTypeNotFoundException if the item ID is not known
   * todo should the exception be checked? wait for usages
   */
  @Nonnull
  String getTypeKey(int id) throws ItemTypeNotFoundException, StructureRuntimeException;

  /**
   * Retrieves a numeric ID for a given item type key. If this is the first time such key is used, an entry in the
   * database is created and ID is assigned. In all later cases, the assigned ID is always returned from this method.
   *
   * @param key item type key, must not be null
   * @return an integer used as the ID of the given item type key
   */
  int getOrCreateTypeId(@Nonnull String key) throws StructureRuntimeException;
}
