package com.almworks.jira.structure.api.attribute.loader.reduce;

import com.almworks.jira.structure.api.attribute.loader.AggregateAttributeLoader;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.List;
import java.util.function.Supplier;

/**
 * <p>
 * Describes how to combine the current value and values aggregated over the part of the subtree
 * (as defined by ReductionStrategy) during aggregate calculation. 
 * It's intended for those attribute loaders that calculate values using only a part of subtree (examples: only direct children, 
 * only leaves, subtree without root node). 
 * The part of subtree to use is defined by {@link ReductionStrategy}, and this interface defines the reduction. 
 * Following how {@link AggregateAttributeLoader AttributeLoader.Aggregate} works, reduction is done by merging the current value with children values.
 * It provides multiple methods for strategies as they can only have a limited set of data on every tree node
 * processing.
 * </p>
 *
 * <p>
 * One would want to implement this interface in case there's a need to aggregate over a custom part of a subtree. 
 * For most cases, it's better to extend {@link ReducingAggregateLoader}.
 * It's sufficient to implement only {@link #reduce}, but one would normally want to implement other methods
 * too -- for either gaining performance benefit (see {@code NumberSumLoader}) or providing more meaningful result.
 * </p>
 *
 * <p>
 * Methods are organized to cover the needs of existing {@link ReductionStrategy} instances.
 * This is the reason methods in {@code ValueReducer} roughly correspond to strategies in {@link ReductionStrategy}.
 * Correspondence table is given below:
 *
 * <table>
 *   <tr><th>Method</th><th>Strategy</th></tr>
 *   <tr><td>{@code reduce(List<T> list)}</td><td>leaves, children, strict</td></tr>
 *   <tr><td>{@code convert(Supplier<T> self)}</td><td>leaves</td></tr>
 *   <tr><td>{@code merge(Supplier<T> self, Supplier<T> reduced)}</td><td>strict</td></tr>
 *   <tr><td>{@code merge(Supplier<T> self, List<T> list)}</td><td>subtree</td></tr>
 * </table>
 *
 * Generally implementations shouldn't worry about tree traversal but can be aware of it.
 * </p>
 *
 * @see ReductionStrategy
 * @see ReducingAggregateLoader
 *
 * @param <T> type of reduced values
 */
@FunctionalInterface
public interface ValueReducer<T> {
  /**
   * Reduces the list of values to one value. Describes 'flat' reduction of elements without hierarchy. Called reducing direct children values or
   * from {@code merge} methods if not overridden. Can be called from {@code convert} method
   *
   * @param list list of values, every item can be null
   * @return value after reduction, null means undefined
   */
  @Nullable
  T reduce(@NotNull List<T> list);

  /**
   * Reduce a single value (self value of attribute for current row). If reduction describes any value conversion (null checks etc) it should be
   * done here as well. The result must be equal to calling reduce() with one element.
   *
   * @param self can return null
   */
  @Nullable
  default T convert(@NotNull Supplier<T> self) {
    return self.get();
  }

  /**
   * Calculates reduction of tree node where {@code self} is value of the current row, and {@code reduced} is result of flat reduction. Value
   * modifications can be done here if one wants to build reduction that computes value based on depth in the structure. Called from strict subtree
   * reduction strategy or from other {@code merge} method if not optimized.
   */
  @Nullable
  default T merge(@NotNull Supplier<T> self, @NotNull Supplier<T> reduced) {
    return reduce(Arrays.asList(self.get(), reduced.get()));
  }

  /**
   * Calculates reduction of tree node where {@code self} is value of the current row and {@code list} represents values passed from children (it
   * may be or be not values of children rows depending on strategy type). Value modifications can be done here if one wants to build reduction
   * that computes value based on depth in the structure. Called from full subtree reduction strategy, that is default and the most frequently
   * used strategy for most reducing aggregates (it makes this method the most probable target for optimization).
   */
  @Nullable
  default T merge(Supplier<T> self, List<T> list) {
    return merge(self, () -> reduce(list));
  }
}
