package com.almworks.jira.structure.api.effector.process;

import com.almworks.jira.structure.api.effector.Effector;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>Represents an {@link Effector} preview, effect application, or undo
 * background process started by a user.
 *
 * <p>Please note that an instance of this class is just a snapshot of
 * the state of the process at the time when the instance itself was created.
 * For those attributes that are supposed to change over time, like
 * {@link #getStatus() status}, calling the corresponding method on the same
 * instance will not return up-to-date values, you must obtain a new
 * {@code EffectorProcess} instance instead.
 */
@PublicApi
public interface EffectorProcess {
  /**
   * @return process ID
   */
  long getId();

  /**
   * Returns the ID of this process' {@link com.almworks.jira.structure.api.process.ProcessHandle
   * ProcessHandle}. The process handle can be used to check the progress
   * of the process.
   *
   * @return process handle ID
   * @see com.almworks.jira.structure.api.process.ProcessHandleManager#getInfo(Long)
   */
  long getProcessHandleId();

  /**
   * Generated effects for preview, not yet applied. Non-empty if
   * {@link EffectorProcess#getStatus()} is one of
   * {@link EffectorProcess.Status#CALCULATED CALCULATED},
   * {@link EffectorProcess.Status#QUEUED QUEUED}, or
   * {@link EffectorProcess.Status#IN_PROGRESS IN_PROGRESS}.
   */
  @Nullable
  EffectorPreview getEffectorPreview();

  /**
   * @return the status of the process
   */
  @NotNull
  Status getStatus();

  /**
   * @return the userkey of the process owner
   */
  @NotNull
  String getUserKey();

  /**
   * @return the timestamp when the process started doing actual work, i.e.
   * transitioned to the {@link Status#CALCULATING CALCULATING} or
   * {@link Status#IN_PROGRESS IN_PROGRESS} status
   */
  @Nullable
  Long getStartTime();

  /**
   * @return the timestamp when the process finished calculating the preview
   * or applying effects.
   */
  @Nullable
  Long getFinishTime();

  /**
   * @return the ID of an earlier effector process whose effects are being
   * reverted by this process, if applicable
   */
  @Nullable
  Long getRevertedProcessId();

  /**
   * @return the "acknowledged" flag for this process
   * @see EffectorProcessManager#acknowledgeFinished(long)
   */
  boolean isAcknowledged();

  /**
   * Represents the status of an effector process.
   */
  enum Status {
    /**
     * Just created, waiting in the queue to begin calculating the preview.
     */
    PENDING,

    /**
     * Currently calculating the preview.
     */
    CALCULATING,

    /**
     * The preview is ready, waiting for confirmation.
     */
    CALCULATED,

    /**
     * Preview calculation failed, effects cannot be applied.
     */
    CALCULATION_FAILED,

    /**
     * Preview calculation was canceled, effects cannot be applied.
     */
    CALCULATION_STOPPED,

    /**
     * Waiting in the queue to begin applying or reverting effects.
     */
    QUEUED,

    /**
     * Currently applying or reverting effects.
     */
    IN_PROGRESS,

    /**
     * Effect application successfully completed, the process is finished.
     */
    COMPLETED,

    /**
     * Effect application completed, but there have been errors. Some effects
     * may have been applied, others failed. The process is finished.
     */
    COMPLETED_WITH_ERRORS,

    /**
     * Effect application was interrupted because Structure plugin was stopped.
     * Some effects may have been applied. The process can be resumed manually.
     */
    APPLY_INTERRUPTED,

    /**
     * Effect application was canceled. Some effects may have been applied.
     * The process is finished.
     */
    APPLY_STOPPED
  }
}
