package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.darkfeature.DarkFeatures;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.*;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>Abstract class for defining a set of items (by their {@link ItemIdentity}).
 * Specific subclasses provide the different ways to define a set.</p>
 *
 * <p>The implementations of this class are immutable and thread-safe.</p>
 *
 * <p>Implementation note: when adding another specific sub-class, make sure it is serializable and can be sent
 * over to the client code.</p>
 *
 * @see TrailItemSet.SpecificItems
 * @see OneType
 */
@PublicApi
public abstract class TrailItemSet {
  @Internal
  @VisibleForTesting
  static int SPECIFIC_ITEMS_LIMIT = DarkFeatures.getInteger("structure.trail.itemLimit", 50);

  @Internal
  @VisibleForTesting
  static int SPECIFIC_TYPES_LIMIT = DarkFeatures.getInteger("structure.trail.typeLimit", 10);

  private int myHashCode;

  /**
   * All specific sub-classes must be declared immediately in this file.
   */
  private TrailItemSet() {
  }

  /**
   * Checks if the set contains the given item.
   *
   * @param id item ID
   * @return true if the item is a part of this set
   */
  public abstract boolean contains(@Nullable ItemIdentity id);

  public boolean containsAny(@Nullable Collection<ItemIdentity> itemIds) {
    if (itemIds == null) return false;
    return itemIds.stream().anyMatch(this::contains);
  }

  /**
   * <p>Expands the set to include the given item. The result of this operation is a new set, which a) includes
   * everything this set includes, b) includes given item.</p>
   *
   * <p>Note that the resulting set may contain <strong>more</strong> items, due to escalation to a more wide
   * set class. If you expand a set by a sufficient number of items, it will switch to be type-based set, which will
   * contain all items of the given types.</p>
   *
   * @param trailItem item to add to the set
   * @return a new set with all items from this set and with {@code trailItem}
   */
  @NotNull
  public abstract TrailItemSet expand(@Nullable ItemIdentity trailItem);

  /**
   * <p>Creates a new set with all items from {@code this} and {@code anotherSet} sets.</p>
   *
   * <p>If {@code anotherSet == null} or {@code this.equals(anotherSet)} - {@code this} set should be returned</p>
   *
   * <p>Note that:</p>
   * <ul>
   * <li>{@code AllItems} set union with any set is always {@code AllItems} set.</li>
   * <li>Type-based set union with any set is type-based set.</li>
   * <li>{@code None} union with non null {@code anotherSet} set is always {@code anotherSet} set.</li>
   * <li>Union of NON type-set may switch to be type-based set if number of items after union is sufficient.</li>
   * <li>Union of type-set may switch to be {@code AllItems} set if number of types after union is sufficient.</li>
   * </ul>
   *
   * @param anotherSet - another set to union with {@code this}
   * @return a new set with all items of {@code this} set and {@code anotherSet}
   */
  @NotNull
  public abstract TrailItemSet union(@Nullable TrailItemSet anotherSet);


  /**
   * Allows the caller to perform per-subclass actions.
   */
  public abstract void accept(@NotNull Visitor visitor);

  /**
   * Returns true if this set will not match any item.
   */
  public final boolean isEmpty() {
    return cardinality() == 0;
  }

  /**
   * <p>Internal method used for implementing equality between different sets.</p>
   *
   * <p>Returns the following number:</p>
   * <ul>
   * <li>0 means empty set.</li>
   * <li>Positive number (but not {@code Integer.MAX_VALUE}) means the number of specific items in the set.</li>
   * <li>Negative number means the set is type-based, and the absolute value means the number of types in the set.</li>
   * <li>{@code Integer.MAX_VALUE} means "all items" set.</li>
   * </ul>
   */
  abstract int cardinality();

  @Override
  public final boolean equals(Object obj) {
    if (!(obj instanceof TrailItemSet)) return false;
    TrailItemSet that = (TrailItemSet) obj;
    if (this.cardinality() != that.cardinality()) return false;
    return Collector.collect(this).equals(Collector.collect(that));
  }

  @Override
  public final int hashCode() {
    int hashCode = myHashCode;
    if (hashCode == 0) {
      hashCode = Collector.collect(this).hashCode();
      if (hashCode == 0) {
        hashCode = -1;
      }
      myHashCode = hashCode;
    }
    return hashCode;
  }

  @Override
  public String toString() {
    return Collector.collect(this).toString();
  }

  /**
   * Constructs a set for specific item IDs.
   *
   * @param ids item identities - null elements are ignored!
   * @return a set that includes at least all the passed items
   */
  public static TrailItemSet of(ItemIdentity... ids) {
    return of(Arrays.asList(ids));
  }

  /**
   * Constructs a set for specific item IDs.
   *
   * @param ids item identities collection - null elements are ignored!
   * @return a set that includes at least all the passed items
   */
  public static TrailItemSet of(Collection<ItemIdentity> ids) {
    Set<ItemIdentity> idsSet = toSetWithoutNulls(ids);
    if (idsSet == null || idsSet.size() == 0) {
      return None.NONE;
    } else if (idsSet.size() == 1) {
      return new OneItem(ids.iterator().next());
    } else if (idsSet.size() <= SPECIFIC_ITEMS_LIMIT) {
      return new SpecificItems(idsSet);
    } else {
      return createFromTypes(converterItemsToItemTypes(idsSet));
    }
  }

  /**
   * Constructs a set for specific item types.
   *
   * @param types item types - null elements are ignored!
   * @return a set that includes all items of the given types
   */
  public static TrailItemSet ofTypes(Collection<String> types) {
    if (types == null) {
      return None.NONE;
    } else {
      return createFromTypes(toSetWithoutNulls(types));
    }
  }

  /**
   * Constructs a some set according to number of types provided.
   *
   * @param typesSet item types - should be cleaned from null elements - it makes no additional check!
   * @return a set that includes all items of the given types
   */
  private static TrailItemSet createFromTypes(@NotNull Set<String> typesSet) {
    if (typesSet.isEmpty()) {
      return None.NONE;
    } else if (typesSet.size() == 1) {
      return new OneType(typesSet.iterator().next());
    } else if (typesSet.size() <= SPECIFIC_TYPES_LIMIT) {
      return new SpecificTypes(typesSet);
    } else {
      return AllItems.ALL_ITEMS;
    }
  }

  private static Set<String> converterItemsToItemTypes(@NotNull Set<ItemIdentity> items) {
    return items.stream()
      .map(id -> id == null ? null : id.getItemType())
      .filter(Objects::nonNull)
      .collect(Collectors.toSet());
  }

  private static <T> Set<T> toSetWithoutNulls(Collection<T> collection) {
    return collection == null ? null : collection.stream().filter(Objects::nonNull).collect(Collectors.toSet());
  }

  static TrailItemSet of(Collection<ItemIdentity> ids, ItemIdentity itemIdentity) {
    ArrayList<ItemIdentity> itemIdentities = new ArrayList<>(ids.size() + 1);
    itemIdentities.addAll(ids);
    itemIdentities.add(itemIdentity);
    return of(itemIdentities);
  }

  static TrailItemSet ofTypes(Collection<String> typesCollectionA, Collection<String> typesCollectionB) {
    return createFromTypes(Stream.concat(typesCollectionA.stream(), typesCollectionB.stream()).collect(Collectors.toSet()));
  }

  static TrailItemSet ofTypes(Collection<String> typesCollectionA, String oneMoreType) {
    return createFromTypes(Stream.concat(typesCollectionA.stream(), Stream.of(oneMoreType)).collect(Collectors.toSet()));
  }

  /**
   * Represents an empty set.
   */
  public static final class None extends TrailItemSet {
    /**
     * An empty set
     */
    public static final TrailItemSet NONE = new None();

    @Override
    public boolean contains(ItemIdentity id) {
      return false;
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null) return this;
      return new OneItem(trailItem);
    }

    @NotNull
    @Override
    public TrailItemSet union(@Nullable TrailItemSet anotherSet) {
      if (anotherSet == null) return this;
      return anotherSet;
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visitNone();
    }

    @Override
    int cardinality() {
      return 0;
    }

    @Override
    public String toString() {
      return "{}";
    }
  }


  /**
   * Represents a set with just one item.
   */
  public static final class OneItem extends TrailItemSet {
    private final ItemIdentity myItem;

    private OneItem(ItemIdentity item) {
      myItem = item;
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return id != null && id.equals(myItem);
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null || trailItem.equals(myItem)) return this;
      return of(myItem, trailItem);
    }

    @NotNull
    @Override
    public TrailItemSet union(@Nullable TrailItemSet anotherSet) {
      if (anotherSet == null) return this;
      return anotherSet.expand(myItem);
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return 1;
    }

    @NotNull
    public ItemIdentity getItem() {
      return myItem;
    }
  }


  /**
   * Represents a set of several specific items.
   */
  public static final class SpecificItems extends TrailItemSet {
    @NotNull
    private final Set<ItemIdentity> myItems;

    private SpecificItems(Set<ItemIdentity> itemsSet) {
      myItems = Collections.unmodifiableSet(itemsSet);
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return myItems.contains(id);
    }

    @NotNull
    public Set<ItemIdentity> getItems() {
      return myItems;
    }

    @Override
    @NotNull
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null || myItems.contains(trailItem)) return this;
      return of(myItems, trailItem);
    }

    @NotNull
    @Override
    public TrailItemSet union(@Nullable TrailItemSet anotherSet) {
      if (anotherSet == null || anotherSet == None.NONE) return this;
      if (anotherSet == AllItems.ALL_ITEMS) {
        return anotherSet;
      }
      if (anotherSet.cardinality() < 0) {
        return ofTypes(Collector.collect(anotherSet).getTypes(), converterItemsToItemTypes(myItems));
      } else {
        Set<ItemIdentity> collectTrailItems = new HashSet<>(myItems);
        collectTrailItems.addAll(Collector.collect(anotherSet).getItems());
        return of(collectTrailItems);
      }
    }

    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return myItems.size();
    }
  }


  /**
   * Represents a set of all items of one specific type.
   */
  public static final class OneType extends TrailItemSet {
    private final String myItemType;

    private OneType(String itemType) {
      myItemType = itemType;
    }

    @NotNull
    public String getItemType() {
      return myItemType;
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return id != null && id.getItemType().equals(myItemType);
    }

    @Override
    @NotNull
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null) return this;
      String itemType = trailItem.getItemType();
      if (itemType.equals(myItemType)) return this;
      return ofTypes(Arrays.asList(myItemType, itemType));
    }

    @NotNull
    @Override
    public TrailItemSet union(@Nullable TrailItemSet anotherSet) {
      if (anotherSet == null || anotherSet == None.NONE) return this;
      if (anotherSet.cardinality() < 0) {
        if (anotherSet instanceof OneType) {
          String otherItemType = ((OneType) anotherSet).myItemType;
          if (myItemType.equals(otherItemType)) {
            return this;
          }
          return ofTypes(Arrays.asList(myItemType, otherItemType));
        }
        return ofTypes(Collector.collect(anotherSet).getTypes(), myItemType);
      }
      return anotherSet.union(this);
    }

    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return -1;
    }
  }


  /**
   * Represents a set of all items of several specific types.
   */
  public static final class SpecificTypes extends TrailItemSet {
    @NotNull
    private final Set<String> myTypes;

    private SpecificTypes(Set<String> types) {
      myTypes = Collections.unmodifiableSet(types);
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return id != null && myTypes.contains(id.getItemType());
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null || myTypes.contains(trailItem.getItemType())) return this;
      return ofTypes(myTypes, trailItem.getItemType());
    }

    @NotNull
    @Override
    public TrailItemSet union(@Nullable TrailItemSet anotherSet) {
      if (anotherSet == null || anotherSet == None.NONE) return this;
      if (anotherSet.cardinality() < 0) {
        if (anotherSet instanceof OneType) {
          String otherItemType = ((OneType) anotherSet).getItemType();
          if (myTypes.contains(otherItemType)) {
            return this;
          }
          return ofTypes(myTypes, otherItemType);
        }
        return ofTypes(myTypes, Collector.collect(anotherSet).getTypes());
      }
      return anotherSet.union(this);
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return -myTypes.size();
    }

    @NotNull
    public Set<String> getTypes() {
      return myTypes;
    }
  }


  /**
   * Represents a set of all items.
   */
  public static final class AllItems extends TrailItemSet {
    public static final TrailItemSet ALL_ITEMS = new AllItems();

    @Override
    public boolean contains(ItemIdentity id) {
      return true;
    }

    @NotNull
    @Override
    public TrailItemSet expand(ItemIdentity trailItem) {
      return this;
    }

    @NotNull
    @Override
    public TrailItemSet union(@Nullable TrailItemSet anotherSet) {
      return this;
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visitAll();
    }

    @Override
    int cardinality() {
      return Integer.MAX_VALUE;
    }
  }


  /**
   * Visitor interface for analyzing the set.
   */
  public interface Visitor {
    void visitNone();

    void visit(OneItem oneItem);

    void visit(SpecificItems set);

    void visit(OneType set);

    void visit(SpecificTypes specificTypes);

    void visitAll();
  }


  /**
   * Alternate visitor interface for reading out the specific items and types.
   */
  public interface ReadVisitor extends Visitor {
    void visitItem(ItemIdentity item);

    void visitType(String type);

    @Override
    default void visitNone() {
    }

    @Override
    default void visit(OneItem set) {
      visitItem(set.getItem());
    }

    @Override
    default void visit(SpecificItems set) {
      for (ItemIdentity item : set.getItems()) {
        visitItem(item);
      }
    }

    @Override
    default void visit(OneType set) {
      visitType(set.getItemType());
    }

    @Override
    default void visit(SpecificTypes set) {
      for (String type : set.getTypes()) {
        visitType(type);
      }
    }
  }


  /**
   * Used to collect specific types and items stored in the TrailItemSet. Not thread-safe.
   */
  public static class Collector implements ReadVisitor {
    private Set<String> myTypes;
    private Set<ItemIdentity> myItems;
    private boolean myAll;

    private Collector(int cardinality) {
      if (cardinality > 0 && cardinality < Integer.MAX_VALUE) {
        myItems = new HashSet<>(cardinality);
      } else if (cardinality < 0) {
        myTypes = new HashSet<>(-cardinality);
      }
    }

    public static Collector collect(TrailItemSet set) {
      Collector collector = new Collector(set.cardinality());
      set.accept(collector);
      return collector;
    }

    public Set<String> getTypes() {
      return myTypes;
    }

    public Set<ItemIdentity> getItems() {
      return myItems;
    }

    public boolean isAll() {
      return myAll;
    }

    @Override
    public void visitItem(ItemIdentity item) {
      if (myItems == null) {
        assert false : "set reported bad cardinality [" + item + "]";
        myItems = new HashSet<>();
      }
      myItems.add(item);
    }

    @Override
    public void visitType(String type) {
      if (myTypes == null) {
        assert false : "set reported bad cardinality [" + type + "]";
        myTypes = new HashSet<>();
      }
      myTypes.add(type);
    }

    @Override
    public void visitAll() {
      myAll = true;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Collector collector = (Collector) o;
      return myAll == collector.myAll &&
        Objects.equals(myTypes, collector.myTypes) &&
        Objects.equals(myItems, collector.myItems);
    }

    @Override
    public int hashCode() {
      return Objects.hash(myTypes, myItems, myAll);
    }

    @Override
    public String toString() {
      if (myAll) return "{ALL}";
      String types = myTypes == null ? "" : StringUtils.join(myTypes, ',');
      String items = myItems == null ? "" : StringUtils.join(myItems, ',');
      String comma = types != null && items != null ? "," : "";
      return "{" + types + comma + items + "}";
    }
  }
}
