package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.*;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Set;

/**
 * <p>An {@code AttributeLoader} contains code that loads values for a particular attribute, represented by {@link AttributeSpec}.
 * Instances of {@code AttributeLoader} are created by {@link AttributeLoaderProvider} in response to a request with
 * a matching spec.</p>
 *
 * <h3>Types of loaders</h3>
 *
 * <p>There are several types of attribute loaders, each represented by a separate interface extending
 * {@code AttributeLoader}:
 *
 * <ul>
 *   <li>{@link ItemAttributeLoader} loads the attribute value for an item (identified by {@link ItemIdentity}),
 *   without any knowledge of the structure / forest that contains this item. These values can be shared between
 *   multiple structures.
 *   </li>
 *
 *   <li>Row attribute loaders use forest information to calculate their value. Therefore, the result provided by a row loader
 *   will be specific to the given forest.
 *   <ul>
 *     <li>{@link SingleRowAttributeLoader} uses only row data for the row that is being calculated.</li>
 *     <li>Multi-row loaders use row data for the row that is being calculated, but also for some additional rows.
 *     <ul>
 *       <li>{@link AggregateAttributeLoader} uses the information calculated for children rows. Sum over a subtree is implemented with an aggregate.</li>
 *       <li>{@link PropagateAttributeLoader} uses the information calculated for the parent row. Inheriting a value from parent is implemented
 *       with a propagate.</li>
 *       <li>{@link ScanningAttributeLoader} uses the information calculated for the row that immediately precedes the current row in the table.
 *       A rolling total is implemented with a scanning loader.</li>
 *     </ul>
 *     </li>
 *   </ul>
 *   </li>
 *
 *   <li>{@link DerivedAttributeLoader} is calculated using only the dependencies. It does not need item or row information.</li>
 * </ul>
 *
 * <p>Every loader must implement one and only one of the interfaces listed above.</p>
 *
 * <h3>Dependencies</h3>
 *
 * <p>Loaders may declare dependencies on other attributes by overriding {@link #getAttributeDependencies()}. To calculate a value, a loader may use the values of dependency attributes.
 * Attribute system guarantees that these values have been calculated and are available.
 * </p>
 *
 * <p>Dependencies are declared as a list of attribute specs. Each spec will be resolved to its own loader. If a spec is not resolved, or if
 * there is a circular dependency, the attribute loader is considered faulty and the values won't be loaded.</p>
 *
 * <h3>State, concurrency, loader caching</h3>
 *
 * <p>An implementation of {@code AttributeLoader} must be stateless. Typically, an instance of loader is created with some parameters
 * and references to the required services. The loader's main loading function may be called concurrently for different or for the same
 * items and rows.</p>
 *
 * <p>An instance of loader is cached and reused to serve multiple requests coming from different users and for different forests. Unless
 * {@link AttributeLoaderProvider} declares the loader non-cacheable, the loader should not assume that the current user at the loader
 * creation time will be the same as when loading function is called.</p>
 *
 * <p>If a loader needs to store some information between the calls to the load function, use {@link AttributeLoaderContext#putObject}
 * and {@link AttributeLoaderContext#getObject}.</p>
 *
 * <h3>Discreteness and purity</h3>
 *
 * <p>When creating a loader for a value that requires doing several things, there's a question of whether to do everything in one
 * loader, or create a chain of loaders with dependencies between them.</p>
 *
 * <p>For example, an attribute that shows the issue release date based on the {@code fixVersions} issue field and the planned release
 * date for those versions, could be implemented in two ways:</p>
 * <ul>
 *   <li>A single {@link ItemAttributeLoader}, which get the value of the issue's {@code fixVersions} and then accesses the versions service
 *   to get the release date.</li>
 *   <li>Two loaders: the first, an {@link ItemAttributeLoader}, loads the versions from {@code fixVersions} field; the second, a
 *   {@link DerivedAttributeLoader}, receives the value from the first loader as a dependency and extracts the release date.</li>
 * </ul>
 *
 * <p>In this example, the second option is clearly better. When deciding which implementation is better, consider the following.</p>
 * <ul>
 *   <li>Can the intermediate result be used as a dependency for loading some other value? And is calculating it relatively costly?
 *   (In our example, {@code fixVersions} field can be used to calculate the names of the versions. Retrieving the field value requires
 *   getting the full Jira issue object, which is costly enough to justify the separation.)</li>
 *   <li>Can the derived result be used with some other intermediate attribute? (In our example, the derived attribute can be used to
 *   calculate the release date based on any other attribute that provides versions, for example, {@code affectsVersions} field.</li>
 * </ul>
 *
 * <p>Multi-row and derived loaders should be separated from getting values related to a particular item or row, which should be done by item
 * or single-row loaders. Multi-row and derived loaders should get input from dependencies and implement only the aggregation/propagation/scanning,
 * so that <em>same inputs will always produce same results</em>. (They can use {@code StructureRow} and {@code ItemIdentity} though -
 * see descriptions of the specific loader interface.)
 * </p>
 *
 * <h3>Context dependencies</h3>
 *
 * <p>If a loader uses some of the context information, such as the current user, it must declare
 * context dependencies by overriding {@link #getContextDependencies()}. The attribute system will then be aware of such dependency and implement correct caching and invalidation
 * for this and all depending attributes.</p>
 *
 * <p><strong>Note:</strong> sometimes the dependency on the user or user's locale is implicit – for example, if you render
 * a web template to get an attribute result, the value would typically depend at least on the current user's locale.</p>
 *
 * <h3>Value caching</h3>
 *
 * <p>One of the most important functions of the attribute subsystem is to cache the calculated values and invalidate / recalculate
 * them when needed. The loader may indicate how the values that it produces should be treated with {@link #getCachingStrategy()}.</p>
 *
 * <h3>Global trail</h3>
 *
 * <p>The global trail allows the loader to declare specific items that, if changed, cause all values calculated for this
 * attribute to be invalidated.</p>
 *
 * <h3>Composite loading and using {@code null} as the result</h3>
 *
 * <p>It's possible to have multiple loaders in the system that load values for the same attribute. This typically means that
 * each loader is able to load the value for specific types of items, or that the attribute was extended at some point to provide
 * (or override) a value in some special case.</p>
 *
 * <p>For example, {@link CoreAttributeSpecs#SUMMARY} is a common attribute that represents an item in the Structure's main column.
 * There are multiple loaders for this attribute, each loading summary for a specific item type only.</p>
 *
 * <p>The common contract for all types of loaders is that returning {@code null} as the result of the calculation means
 * <em>"I cannot work with this item/row, let some other loader try"</em>. However, if the loader is indeed responsible for
 * this item or row, but the value is missing or cannot be calculated for some reason, the loader should
 * return {@link AttributeValue#undefined()}.</p>
 *
 * <p>Since the system will try all loaders in sequence until some loader returns a non-null value, the order of loaders is important.
 * See {@link AttributeLoaderProvider} about how to define in what order the attribute loaders are called.
 * (If you initialize {@link com.almworks.jira.structure.api.attribute.loader.composition.CompositeAttributeLoader#create(AttributeSpec, List)} manually, the passed loaders will be invoked in the iteration order.)
 * </p>
 *
 * <h3>Reentrancy</h3>
 *
 * <p>The loaders are not supposed to interact with Attribute and Forest subsystems as client code. In particular, calling
 * {@link StructureAttributeService#getAttributeValues} or {@link com.almworks.jira.structure.api.forest.ForestService#getForestSource} from within a loading function
 * may lead to re-entrancy issues and unexpected results, exceptions or endless cycle.</p>
 *
 * <h3>Preloading</h3>
 *
 * <p>Typically, attribute system receives requests to load attributes for a number of rows or items. It may be more efficient for a
 * loader to perform some bulk action before per-item or per-row loading function is called. This is called preloading and both item
 * and row-based loaders support it.</p>
 *
 * @param <T> type of the loaded value
 * @see AttributeLoaderProvider
 * @see ItemAttributeLoader
 * @see DerivedAttributeLoader
 * @see SingleRowAttributeLoader
 * @see AggregateAttributeLoader
 * @see PropagateAttributeLoader
 * @see ScanningAttributeLoader
 */
@PublicSpi
public interface AttributeLoader<T> {
  /**
   * <p>Returns the spec for which this loader loads values.</p>
   * <p>The returned value must be the same throughout the lifetime of the object.</p>
   *
   * @return spec being loaded
   */
  @NotNull
  AttributeSpec<T> getAttributeSpec();

  /**
   * <p>Returns attributes that need to be loaded prior to calling this loader's loading function.</p>
   * <p>The returned value must be the same throughout the lifetime of the object.</p>
   *
   * @return the set of attribute dependencies, or {@code null} if there are none
   *
   * @see AttributeLoaderContext#getDependencyValue
   */
  @Nullable
  default Set<AttributeSpec<?>> getAttributeDependencies() {
    return null;
  }

  /**
   * <p>Indicates which context values are used to calculate the value of the attribute.</p>
   *
   * <p>For example, if the calculated value depends on the current user, the return value from this method must include
   * {@link AttributeContextDependency#USER}.</p>
   *
   * <p>The returned value must be the same throughout the lifetime of the object.</p>
   *
   * @return the set of context dependencies or {@code null} if there are none
   */
  @Nullable
  default Set<AttributeContextDependency> getContextDependencies() {
    return AttributeContextDependency.guessContextDependencies(getAttributeSpec());
  }

  /**
   * <p>Indicates how the values provided by this loader should be cached.</p>
   * <p>The returned value must be the same throughout the lifetime of the object.</p>
   *
   * @return caching strategy, {@code null} has the same effect as {@link AttributeCachingStrategy#MAY}
   */
  @Nullable
  default AttributeCachingStrategy getCachingStrategy() {
    return AttributeCachingStrategy.guessCachingStrategy(getAttributeSpec());
  }

  /**
   * <p>Returns global trail. When an item from a global trail changes, all values calculated for this attribute will be
   * invalidated and will need to be recalculated.</p>
   *
   * <p>The returned value must be the same throughout the lifetime of the object.</p>
   *
   * @return global trail set, or {@code null} if none
   */
  @Nullable
  default TrailItemSet getGlobalTrail() {
    return null;
  }
}
