package com.almworks.jira.structure.api.structure.history;

import com.almworks.integers.LongList;
import com.almworks.jira.structure.api.forest.raw.Forest;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Objects;

public final class HistoryEntry {
  private final long myStructure;
  private final int myVersion;
  private final int myPrevVersion;
  private final long myTimestamp;
  private final String myUserKey;

  @Nullable
  private final Long mySynchronizer;

  @NotNull
  private final List<Change> myChanges;

  /**
   * @param changes immutable list of changes
   */
  public HistoryEntry(long structure, int version, int prevVersion, long timestamp, String userKey, @Nullable Long synchronizer, @NotNull List<Change> changes) {
    myStructure = structure;
    myVersion = version;
    myPrevVersion = prevVersion;
    myTimestamp = timestamp;
    myUserKey = StringUtils.isEmpty(userKey) ? null : userKey;
    mySynchronizer = synchronizer;
    myChanges = changes;
  }

  public long getStructure() {
    return myStructure;
  }

  public int getVersion() {
    return myVersion;
  }

  public int getPrevVersion() {
    return myPrevVersion;
  }

  public long getTimestamp() {
    return myTimestamp;
  }

  public String getUserKey() {
    return myUserKey;
  }

  @Nullable
  public Long getSynchronizer() {
    return mySynchronizer;
  }

  @NotNull
  public List<Change> getChanges() {
    return myChanges;
  }

  @Override
  public String toString() {
    return "HistoryEntry{" +
      "myStructure=" + myStructure +
      ", myVersion=" + myVersion +
      ", myPrevVersion=" + myPrevVersion +
      ", myTimestamp=" + myTimestamp +
      ", myUserKey='" + myUserKey + '\'' +
      ", mySynchronizer=" + mySynchronizer +
      ", myChanges=" + myChanges +
      '}';
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HistoryEntry that = (HistoryEntry) o;
    return myStructure == that.myStructure &&
      myVersion == that.myVersion &&
      myPrevVersion == that.myPrevVersion &&
      myTimestamp == that.myTimestamp &&
      Objects.equals(myUserKey, that.myUserKey) &&
      Objects.equals(mySynchronizer, that.mySynchronizer) &&
      Objects.equals(myChanges, that.myChanges);
  }

  @Override
  public int hashCode() {
    return Objects.hash(myStructure, myVersion, myPrevVersion, myTimestamp, myUserKey, mySynchronizer, myChanges);
  }


  public enum Operation {
    ADD("add"), COPY("copy"), MOVE("move"), REMOVE("remove");

    private final String myExternalName;

    Operation(String externalName) {
      myExternalName = externalName;
    }

    public String getExternalName() {
      return myExternalName;
    }

    public static Operation forExternalName(String externalName) {
      for (Operation type : values()) {
        if (type.myExternalName.equals(externalName)) {
          return type;
        }
      }
      return null;
    }
  }


  public static final class Change {
    private final Operation myOperation;
    private final Forest myForest;
    private final LongList myPathFrom;
    private final LongList myAfterFrom;
    private final LongList myPathTo;
    private final LongList myAfterTo;
    private final int myDirection;
    private final LongList myOriginalRows;

    public Change(Operation operation, Forest forest, LongList pathFrom, LongList afterFrom, 
      LongList pathTo, LongList afterTo, int direction, LongList originalRows) 
    {
      myOperation = operation;
      myForest = forest;
      myPathFrom = pathFrom;
      myAfterFrom = afterFrom;
      myPathTo = pathTo;
      myAfterTo = afterTo;
      myDirection = direction;
      myOriginalRows = originalRows;
    }

    public Operation getOperation() {
      return myOperation;
    }

    public Forest getForest() {
      return myForest;
    }

    public LongList getPathFrom() {
      return myPathFrom;
    }

    public LongList getAfterFrom() {
      return myAfterFrom;
    }

    public LongList getPathTo() {
      return myPathTo;
    }

    public LongList getAfterTo() {
      return myAfterTo;
    }

    public int getDirection() {
      return myDirection;
    }

    public LongList getOriginalRows() {
      return myOriginalRows;
    }

    @Override
    public String toString() {
      return "Change{" +
        "myOperation=" + myOperation +
        ", myForest=" + myForest +
        ", myPathFrom=" + myPathFrom +
        ", myAfterFrom=" + myAfterFrom +
        ", myPathTo=" + myPathTo +
        ", myAfterTo=" + myAfterTo +
        ", myDirection=" + myDirection +
        ", myOriginalRows=" + myOriginalRows +
        '}';
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Change change = (Change) o;
      return myDirection == change.myDirection &&
        myOperation == change.myOperation &&
        Objects.equals(myForest, change.myForest) &&
        Objects.equals(myPathFrom, change.myPathFrom) &&
        Objects.equals(myAfterFrom, change.myAfterFrom) &&
        Objects.equals(myPathTo, change.myPathTo) &&
        Objects.equals(myAfterTo, change.myAfterTo) &&
        Objects.equals(myOriginalRows, change.myOriginalRows);
    }

    @Override
    public int hashCode() {
      return Objects.hash(myOperation, myForest, myPathFrom, myAfterFrom, myPathTo, myAfterTo, myDirection, myOriginalRows);
    }
  }
}
