package com.almworks.jira.structure.api.row;

import com.almworks.jira.structure.api.attribute.StructureAttributeService;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.Internal;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>The super-root is a fictional row that is present in every forest as the parent of all root rows.
 * {@code SuperRootRow} class is the representation of the super-root row.</p>
 *
 * <p>Super-root is a concept with limited application; as of now, it can only be used in {@link StructureAttributeService} for calculating totals
 * for the whole forest.</p>
 *
 * <p>The super-root row has row ID of {@code -1}. This row ID is reserved and must not be used for any other purpose. Creating a {@link Forest}
 * with row ID {@code -1} will result in exception.</p>
 *
 * @see StructureAttributeService
 */
public final class SuperRootRow implements StructureRow {
  /**
   * Super-root's row ID. This ID is reserved and may not be used for any other rows.
   */
  public static final long SUPER_ROOT_ROW_ID = -1;

  /**
   * A single instance of the super-root {@link StructureRow}.
   */
  public static final SuperRootRow SUPER_ROOT_ROW = new SuperRootRow();

  /**
   * A special item type for the super-root item.
   */
  public static final String SUPER_ROOT_ITEM_TYPE = "super-root";

  /**
   * An {@link ItemIdentity} of the item in the super-root.
   */
  public static final ItemIdentity SUPER_ROOT_ITEM = ItemIdentity.longId(SUPER_ROOT_ITEM_TYPE, -1L);

  /**
   * A special semantics of the super-root.
   */
  public static final long SUPER_ROOT_SEMANTICS = -1;

  /**
   * Super-root's index in an array of forest rows. Since it is a fictional row, the index is also fictional. Don't use it outside the attribute system.
   */
  @Internal
  public static final int SUPER_ROOT_INDEX = -1;

  /**
   * Super-root's depth in a forest. Since it is a fictional row, the depth is also fictional. Don't use it outside the attribute system.
   */
  @Internal
  public static final int SUPER_ROOT_DEPTH = -1;

  private SuperRootRow() {}

  @Override
  public long getRowId() {
    return SUPER_ROOT_ROW_ID;
  }

  @NotNull
  @Override
  public ItemIdentity getItemId() {
    return SUPER_ROOT_ITEM;
  }

  @Override
  public long getSemantics() {
    return SUPER_ROOT_SEMANTICS;
  }

  @Nullable
  @Override
  public <I> I getItem(@NotNull Class<I> itemClass) {
    return null;
  }

  @Override
  public String toString() {
    return "super-root";
  }
}
