package com.almworks.jira.structure.api.forest.raw;

/**
 * This interface may be used to collect additional information while forest changes are being made. An instance of
 * <code>ForestChangeEventHandler</code> is passed to one of the mutator methods of the <code>Forest</code>
 * instance like {@link Forest#moveSubtree(long, long, long, ForestChangeEventHandler)} and corresponding methods
 * are called before or after the change is made to the forest.
 *
 * @author Igor Sereda
 */
public interface ForestChangeEventHandler {
  /**
   * <p>Called at the moment before a subtree is removed from the forest.</p>
   *
   * <p>The code must not modify the forests, otherwise the result is undefined.</p>
   *
   * @param forest the forest being changed
   * @param fromIndex the index of the root element of the subtree
   * @param toIndex the index of an element that follows the removed subtree (or the size of the forest if the subtree is
   * at the end of the forest). Invariant: <code>toIndex == forest.getSubtreeEnd(fromIndex)</code>
   * @param removedSubtree the copy of the subtree being removed
   */
  void beforeSubtreeRemoved(Forest forest, int fromIndex, int toIndex, Forest removedSubtree);

  /**
   * <p>Called at the moment after another forest is added to the forest being observed. The added forest
   * completely resides under a certain parent issue, or at the top level. In other words, it is impossible
   * that any issue within range <code>[fromIndex, toIndex)</code> has depth less than the first issue of the added forest <code>[fromIndex]</code>.</p>
   *
   * <p>The code must not modify the forests, otherwise the result is undefined.</p>
   *
   * @param forest the forest being changed
   * @param fromIndex the index of the first element being inserted
   * @param toIndex the index of an element that follows the added part of the forest (or the size of the forest if the issues
   * are added at the end).
   * @param insertedForest the forest that has been inserted
   */
  void afterForestInserted(Forest forest, int fromIndex, int toIndex, Forest insertedForest);
}
