package com.almworks.jira.structure.api.export;

import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;

/**
 * <p>An {@code ExportRenderer} corresponds to a single column of the underlying table-like medium (e.g. an Excel sheet
 * or an HTML table). The renderer is responsible for configuring the column and rendering its cells.</p>
 *
 * <p>Single-format renderers can be parameterized with the expected {@link ExportCell} and {@link ExportColumn}
 * subtypes. Generic or multiple-format renderers can use the basic cell and column types or implement the raw
 * interface.</p>
 *
 * @param <C> The expected cell type.
 * @param <M> The expected column type.
 *
 * @see ExportCell
 * @see ExportColumn
 */
@PublicSpi
public interface ExportRenderer<C extends ExportCell, M extends ExportColumn<C>> {
  /**
   * Allows the renderer to do one-time preparations before rendering a forest.
   * The renderer should request attributes that it needs for rendering through
   * {@link ExportRequestContext#requireAttribute}.
   *
   * @param context rendering context
   */
  void prepare(@NotNull ExportRequestContext context);

  /**
   * Configure the column, i.e. set the column name (through the {@link ExportColumn#getHeaderCell() header cell})
   * and, probably, other column options, depending on the medium. This method is called once for export request.
   *
   * @param column The column instance to configure.
   * @param context The current request context.
   */
  void configureColumn(@NotNull M column, @NotNull ExportRenderContext context);

  /**
   * Render the cell, i.e. set its value and, probably, other cell options, depending on the medium. This method is
   * called once for each issue in the exported structure (or part of a structure).
   *
   * @param cell The cell instance to render into.
   * @param row The current row.
   * @param context The current request context.
   */
  void renderCell(@NotNull C cell, @NotNull ExportRow row, @NotNull ExportRenderContext context);
}
