package com.almworks.jira.structure.api.event;

import com.almworks.integers.LongList;
import org.jetbrains.annotations.Nullable;

/**
 * <p><code>IssueEventBridge</code> is a simple hub that allows to listen to issue changes
 * and report issue changes.</p>
 *
 * <p>The bridge subscribes to the JIRA notification system to get the standard notifications
 * about the issue changes. Additionally, it allows components to report issue changes.</p>
 *
 * <p>Only the IDs of the issues that have changed are reported, not the changes themselves.
 * The bridge is also allowed to delay reporting changes a bit to accumulate a bigger change set.</p>
 *
 * @author Igor Sereda
 */
public interface IssueEventBridge {
  /**
   * Add an <code>IssueListener</code> that will get notifications of the subsequent changes.
   *
   * @param listener the listener
   */
  public void addListener(@Nullable IssueListener listener);

  /**
   * Removes the listener from the component
   *
   * @param listener the listener
   */
  public void removeListener(@Nullable IssueListener listener);

  /**
   * This method is used to notify the bridge that the listed issues have changed or that the listeners must
   * receive the notification anyway, probably to recount something.
   *
   * @param issueIds the list of changed issues
   * @param eventType the event type
   */
  void reportChanges(@Nullable LongList issueIds, JiraChangeType eventType);

  /**
   * This method is used to notify the bridge about a specifically constructed event object, which gets
   * rebroadcast to the listeners.
   *
   * @param event the event describing a change
   */
  void reportEvent(@Nullable JiraChangeEvent event);
}
