package com.almworks.jira.structure.api.error;

import com.almworks.jira.structure.api.item.CoreIdentities;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.annotation.*;

/**
 * <p>{@code StructureError} implementations are wrappers around numeric error codes, which also
 * provide the error's category (see {@link StructureErrorCategory}) and error name.
 * This lets the client code distinguish between multiple cases where
 * {@code StructureException} can be thrown.</p>
 *
 * <p>{@code StructureError} may also be used to conveniently start a construction of {@link StructureException}.</p>
 *
 * <p>All public error codes are listed in {@code StructureErrors}.</p>
 *
 * <h3>Implementation Notes</h3>
 *
 * <p>{@code StructureError} is an interface to allow addition of new error codes without the need
 * to adjust previously released classes. Structure plugin may have unpublished internal error codes
 * and Structure extensions may have a collection of their own.</p>
 *
 * <p>When assigning numeric value to a new error code, please mind that numbers 0&mdash;9999 are reserved
 * by ALM Works for Structure and Structure extensions.</p>
 *
 * @see StructureException
 * @see StructureErrors
 */
@PublicApi
@PublicSpi
public interface StructureError {
  /**
   * Returns error code.
   */
  int getCode();

  /**
   * Returns error category.
   */
  StructureErrorCategory getCategory();

  /**
   * Returns error name. The name is not supposed to tell end-user anything, it is not i18n-ized and
   * can only be written as a diagnostic information.
   */
  @NotNull
  default String name() {
    return "error#" + getCode();
  }

  /**
   * Checks if the error is of the given category.
   *
   * @param category the category
   * @return true if this error belongs to the category
   */
  default boolean is(StructureErrorCategory category) {
    return getCategory() == category;
  }

  /**
   * Checks if the error belongs to one of the given categories.
   *
   * @param categories categories
   * @return true if this error belongs to one of the categories
   */
  default boolean isOneOf(StructureErrorCategory ... categories) {
    StructureErrorCategory c = getCategory();
    for (StructureErrorCategory category : categories) {
      if (c == category) {
        return true;
      }
    }
    return false;
  }

  /**
   * Checks if the error is actually one of the provided variants.
   *
   * @param errors errors to compare to
   * @return true if this error is equal to one of the passed errors
   */
  default boolean isOneOf(StructureError ... errors) {
    for (StructureError error : errors) {
      if (equals(error)) {
        return true;
      }
    }
    return false;
  }


  /**
   * Creates a new builder for {@link StructureException} with this code.
   *
   * @return {@code StructureException} builder
   */
  @NotNull
  default StructureException.Builder builder() {
    return new StructureException.Builder(this);
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param messageKey i18n key for the localized error message
   * @param messageParameters optional i18n message parameters
   * @return exception with this code ready to be thrown
   */
  @NotNull
  default StructureException withLocalizedMessage(@NotNull String messageKey, Object... messageParameters) {
    return builder().withLocalizedMessage(messageKey, messageParameters);
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param message non-internationalized message
   * @return exception with this code  ready to be thrown
   */
  @NotNull
  default StructureException withMessage(@Nullable String message) {
    return builder().withMessage(message);
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @return exception with this code  ready to be thrown
   */
  @NotNull
  default StructureException withoutMessage() {
    return builder().withoutMessage();
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param structure related structure ID
   * @return {@code StructureException} builder
   */
  @NotNull
  default StructureException.Builder forStructure(@Nullable Long structure) {
    return builder().forStructure(structure);
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param view related view ID
   * @return {@code StructureException} builder
   */
  @NotNull
  default StructureException.Builder forView(@Nullable Long view) {
    return builder().forView(view);
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param row related row ID
   * @return {@code StructureException} builder
   */
  @NotNull
  default StructureException.Builder forRow(@Nullable Long row) {
    return builder().forRow(row);
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param item related item ID
   * @return {@code StructureException} builder
   */
  @NotNull
  default StructureException.Builder forItem(@Nullable ItemIdentity item) {
    return builder().forItem(item);
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param issueId related issue ID
   * @return {@code StructureException} builder
   */
  @NotNull
  default StructureException.Builder forIssue(long issueId) {
    return forItem(CoreIdentities.issue(issueId));
  }

  /**
   * A shortcut to calling the same method on {@link StructureException} builder.
   *
   * @param cause an exception that caused to problem
   * @return {@code StructureException} builder
   */
  @NotNull
  default StructureException.Builder causedBy(@Nullable Throwable cause) {
    return builder().causedBy(cause);
  }


  /**
   * Auxiliary annotation, which may be used on a collection of {@link StructureError} instances to specify the
   * inclusive range of error codes that is taken by that collection.
   */
  @Target(ElementType.TYPE)
  @Retention(RetentionPolicy.RUNTIME)
  @interface CodeRange {
    int from();
    int to();
  }
}
