package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.ValueFormat;
import com.atlassian.annotations.PublicApi;

/**
 * <p>Defines how the values provided by an attribute loader are cached.</p>
 *
 * <p>Most attribute loaders should not need to define the caching strategy, as the system will try to pick the optimal strategy. By default,
 * attributes engine tries to cache most of the values.</p>
 *
 * @see AttributeLoader#getCachingStrategy()
 */
@PublicApi
public enum AttributeCachingStrategy {
  /**
   * <p>The value should be cached to avoid excessive recalculation. This has roughly the same effect as {@link #MAY}.</p>
   *
   * <p>One notable distinction is that if an attribute loader declares that the values should be cached, but then they are not cached
   * (for example, if the values depend on other values, which must not be cached), the system will issue a warning in the logs.</p>
   */
  SHOULD,

  /**
   * The value may be cached. The system will store the value in a cache unless there are other indications that it shouldn't.
   */
  MAY,

  /**
   * <p>The value should not be cached, although it potentially could be done.</p>
   *
   * <p>There are two typical cases:</p>
   * <ul>
   *   <li>The value is a Java object that may become outdated - for example, {@code Project} object or any other bean-type value holder;</li>
   *   <li>The value is very easy to calculate and it would be a waste of memory to store it.</li>
   * </ul>
   *
   * <h3>Strategy propagation to dependent loaders</h3>
   *
   * <p>Unlike {@link #MUST_NOT}, this caching strategy does not affect caching strategies of the dependent
   * attributes. If a value should not be cached (and is not cached), the derived values may still be cached if they have corresponding
   * caching strategies.</p>
   *
   * <p>This means that the attribute loader using this caching strategies must not use some untraceable data when calculating the value. All
   * dependent values will have to be invalidated with the usual updates to the item and, in case of multi-row loaders, the corresponding
   * forest dependencies.</p>
   */
  SHOULD_NOT,

  /**
   * <p>The value must not be cached. Use this strategy to indicate that the value is based on some untraceable data, so the attribute system
   * will not know when it's time to invalidate and recalculate the value.</p>
   *
   * <h3>Strategy propagation to dependent loaders</h3>
   *
   * <p>Declaring that the system must not cache the values coming from an attribute loader will also make values from all dependent loaders
   * non-cacheable.</p>
   *
   * <p>When strategies need to be combined (for example, when figuring out the final strategy based on dependencies), {@code MUST_NOT} strategy
   * overrules any other strategy, because otherwise the values served might not be correct.</p>
   *
   * <h3>Performance implications</h3>
   *
   * <p><strong>Try to avoid using this strategy!</strong> It may have adverse effect on performance, since the system will need to recalculate
   * the values not only for this attribute, but for any dependent attributes too.</p>
   *
   * <p>Consider alternatives to declaring the values to be non-cacheable, for example, setting a value timeout - see {@link AttributeLoaderContext#valueExpires}.</p>
   */
  MUST_NOT;

  /**
   * Guessing if the value is cacheable, based on the value format. A non-standard format will likely be an object
   * like Project, which shouldn't be cacheable.
   */
  public static AttributeCachingStrategy guessCachingStrategy(AttributeSpec<?> spec) {
    ValueFormat<?> format = spec.getFormat();
    if (ValueFormat.getStandardFormat(format.getFormatId()) != null) {
      return MAY;
    } else {
      //we guess that it's an internal Jira object (User/Project/...)
      return SHOULD_NOT;
    }
  }
}
