package com.almworks.jira.structure.api.structure;

import com.almworks.jira.structure.api.auth.StructureAuth;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.ForestService;
import com.almworks.jira.structure.api.forest.ForestSource;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.permissions.PermissionLevel;
import com.almworks.jira.structure.api.util.CallableE;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * <p>
 * <code>StructureManager</code> is the main component used to manage structures as entities.
 * To access and manage structure contents - its {@link Forest forest}, a hierarchical list of items, - 
 * see {@link ForestService}. 
 * </p>
 * <p>
 * A {@link Structure} is a named entity that has properties like name, description and permissions. 
 * Methods in this interface allow creating/removing structures and changing those properties.
 * </p>
 * <p>
 * Unless otherwise stated, all methods are thread-safe.
 * Note that the methods of the {@link Structure} interface are not thread-safe - you need
 * to retrieve a separate instance in every thread where you need it. </p>
 * <p>
 * All methods that return <code>Structure</code> instances return new independent instances every time they are
 * called.
 * </p>
 * <p>
 * All methods check permissions for the current user as returned by {@link StructureAuth#getUser()}, unless
 * permission checking is turned off by setting {@link StructureAuth#isSecurityOverridden()} flag
 * (useful if you are writing an administrative task - see {@link StructureAuth#sudo(ApplicationUser, boolean, CallableE)}).
 * </p>
 * <h2>Examples</h2>
 * <p>
 * Creating a structure:
 * </p>
 * <pre>
 *   Structure newStructure = 
 *     structureManager.createStructure().setName("structureName").saveChanges();
 * </pre>
 * <p>
 * Changing a structure:
 * </p>
 * <pre>
 *   List&lt;Structure&gt; structures =
 *     structureManager.getStructuresByName("hideme", ADMIN);
 *   for (Structure s : structures) {
 *     s.setPermissions(Collections.emptyList()).saveChanges();
 *   }
 * </pre>
 *
 * @author Igor Sereda
 * @see Structure
 * @see Forest
 * @see ForestService
 */
@PublicApi
public interface StructureManager {
  /**
   * <p>Retrieves a structure given structure ID. 
   * Throws exception if the structure does not exist or is not accessible.</p>
   * 
   * <p>The second parameter allows to specify the required permission level for the current user.</p>
   * 
   * <p>Unless {@link StructureAuth#isSecurityOverridden()} is true, we also check if the user 
   * has access to Structure plugin.</p>
   *
   * <p>Every call to {@code getStructure()} will return a new instance of {@code Structure}, which will contain
   * a snapshot of the structure's properties. You shouldn't keep that instance around for long, because any
   * changes to the structure, made by other code in parallel, will not be reflected in this snapshot instance.
   * </p>
   *
   * @param structureId the ID of the structure, nullable for convenience (if null, will throw StructureException)
   * @param requiredLevel when checking for user's permission, require that the user has at least the specified permission.
   *   Passing null of {@link PermissionLevel#NONE} effectively disables the structure permissions check, but we still
   *   check that the user has access to Structure plugin (unless {@link StructureAuth#isSecurityOverridden()} is true).
   * @return an instance of a structure, not null
   * @throws StructureException if structure is not found, or the user is not allowed to use Structure,
   * or the user does not have the required access level to this structure, or if any other problem is encountered
   */
  @NotNull
  Structure getStructure(@Nullable Long structureId, @Nullable PermissionLevel requiredLevel) throws StructureException;

  /**
   * <p>Retrieves a list of all unarchived structures that are accessible to the 
   * current user at the specified permission level.</p>
   * 
   * <p>The method scans all existing structures, selects unarchived ones and checks permissions.
   * If the user has no access to Structure plugin, returns empty list.
   * If {@link StructureAuth#isSecurityOverridden()} is true, ignores all of the checks above and 
   * returns all unarchived structures.</p>
   * 
   * <p>The resulting list is sorted by name, using the current user's locale.</p>
   *
   * @param requiredLevel when checking for user's permission, require that the user 
   *   has at least the specified permission for every structure in the returned list.
   *   Passing null of {@link PermissionLevel#NONE} effectively disables the structure permissions check, but we still 
   *   check that the user has access to Structure plugin (unless {@link StructureAuth#isSecurityOverridden()} is true).
   * @return a list of unarchived structures available to the user, not null
   * @see #getAllStructures(PermissionLevel, boolean) 
   */
  @NotNull
  List<Structure> getAllStructures(@Nullable PermissionLevel requiredLevel);

  /**
   * <p>Retrieves a list of all structures that are visible to the current user 
   * at the specified permission level.</p>
   * 
   * <p>The method scans all existing structures and checks permissions.
   * If the user has no access to Structure plugin, returns empty list.
   * If {@link StructureAuth#isSecurityOverridden()} is true, ignores all of the checks above and returns all structures
   * (or only unarchived, depending on the parameter).</p>
   * 
   * <p>The resulting list is sorted by name, using the current user's locale.</p>
   *
   * @param requiredLevel when checking for user's permission, require that the user 
   *   has at least the specified permission for every structure in the returned list.
   *   Passing null of {@link PermissionLevel#NONE} effectively disables the structure permissions check, but we still 
   *   check that the user has access to Structure plugin (unless {@link StructureAuth#isSecurityOverridden()} is true).
   * @param includeArchived if true, result will also contain archived structures
   * @return a list of structures available to the user, not null
   */
  @NotNull
  List<Structure> getAllStructures(@Nullable PermissionLevel requiredLevel, boolean includeArchived);

  /**
   * <p>Retrieves a list of all archived structures that are visible to the current user
   * at the specified permission level.</p>
   * 
   * <p>The method scans all existing structures, selects archived ones and checks permissions.
   * If the user has no access to Structure plugin, returns empty list.
   * If {@link StructureAuth#isSecurityOverridden()} is true, ignores all of the checks above and 
   * returns all archived structures.</p>
   *
   * <p>The resulting list is sorted by name, using the current user's locale.</p>
   * 
   * @param requiredLevel when checking for user's permission, require that the user 
   *   has at least the specified permission for every structure in the returned list.
   *   Passing null of {@link PermissionLevel#NONE} effectively disables the structure permissions check, but we still 
   *   check that the user has access to Structure plugin (unless {@link StructureAuth#isSecurityOverridden()} is true).
   * @return a list of archived structures available to the user, not null
   * */
  @NotNull
  List<Structure> getArchivedStructures(@Nullable PermissionLevel requiredLevel);

  /**
   * <p>Retrieves a list of all structures visible to the current user,
   * for which the specified user is the owner.</p>
   * 
   * <p>The method scans all existing structures and looks for those that are 
   * owned by the specified user and visible to the current user.
   * If the user has no access to Structure plugin, returns empty list.
   * If {@link StructureAuth#isSecurityOverridden()} is true, ignores all of the checks above and returns 
   * all structures that the specified user owns.</p>
   * 
   * <p>The resulting list is sorted by name, using the current user's locale.</p>
   * 
   * @param user owner of the structures to look for
   * @return a list of structures owned by the specified user, not null
   * @see Structure#getOwner
   * */
  @NotNull
  List<Structure> getStructuresOwnedBy(@Nullable ApplicationUser user);

  /**
   * <p>Convenience method to search for unarchived structures with the specified name. Using this method is more
   * efficient than retrieving all structures and filtering.</p>
   * 
   * <p>Structures are not required to have unique names, therefore this method returns a list.</p>
   * 
   * <p>Returns only structures for which the current user has access at 
   * the specified permission level.
   * If the user has no access to Structure plugin, returns empty list.
   * If {@link StructureAuth#isSecurityOverridden()} is true, ignores all of the checks above.</p>
   * 
   * <p>The resulting list is sorted by name, using the current user's locale.</p>
   *
   * @param name the name of the structure sought, <strong>case-insensitive</strong> and <strong>trim-insensitive</strong> 
   *   (leading and trailing whitespace don't matter). If <code>null</code>, an empty list is returned.
   * @param requiredLevel when checking for user's permission, require that the user has at least the specified permission 
   *   for every structure in the returned list.
   *   Passing null or {@link PermissionLevel#NONE} effectively disables the structure permissions check, but we still
   *   check that the user has access to Structure plugin (unless {@link StructureAuth#isSecurityOverridden()} is true).
   * @return a list of the matching unarchived structures
   */
  @NotNull
  List<Structure> getStructuresByName(@Nullable String name, @Nullable PermissionLevel requiredLevel);

  /**
   * <p>Convenience method that searches for structures with the specified name. 
   * It is more efficient than retrieving all structures and filtering.</p>
   *
   * <p>Structures are not required to have unique names, therefore this method returns a list.</p>
   *
   * <p>Returns only structures for which the current user has access at 
   * the specified permission level.
   * If the user has no access to Structure plugin, returns empty list.
   * If {@link StructureAuth#isSecurityOverridden()} is true, ignores all of the checks above.</p>
   *
   * <p>The resulting list is sorted by name, using the current user's locale.</p>
   *
   * @param name the name of the structure sought, <strong>case-insensitive</strong> and <strong>trim-insensitive</strong> 
   *   (leading and trailing whitespace don't matter). If <code>null</code>, an empty list is returned.
   * @param requiredLevel when checking for user's permission, require that the user has at least the specified permission 
   *   for every structure in the returned list.
   *   Passing null or {@link PermissionLevel#NONE} effectively disables the structure permissions check, but we still
   *   check that the user has access to Structure plugin (unless {@link StructureAuth#isSecurityOverridden()} is true).
   * @param searchArchivedStructures if true, result will also contain archived structures
   * @return a list of the matching unarchived structures
   */
  @NotNull
  List<Structure> getStructuresByName(@Nullable String name, @Nullable PermissionLevel requiredLevel,
    boolean searchArchivedStructures);

  /**
   * <p>Convenience method that checks whether the current user can see at least one
   * unarchived structure. This works functionally the same as 
   * <code>!getAllStructures(PermissionLevel.VIEW).isEmpty()</code>, 
   * but much faster.</p>
   * 
   * <p>If the user has no access to Structure plugin, returns false. 
   * (Unless {@link StructureAuth#isSecurityOverridden()} is on, in which case user's access to Structure is not
   * checked.)</p>
   * 
   * @return true if the current user has access to the Structure plugin
   *   (this check is skipped if {@link StructureAuth#isSecurityOverridden()} is true) 
   *   and if there exists at least one unarchived structure, to which the current user has {@link PermissionLevel#VIEW} access.
   */
  boolean hasNonArchivedStructuresForUser();

  /**
   * <p>Convenience method that returns the ID of a single structure that the 
   * current user can see. 
   * This works much faster than <code>getAllStructures(PermissionLevel.VIEW)</code>.</p>
   *
   * <p>If the user does not have {@link PermissionLevel#VIEW} access to any structure, 
   * or if the user has the access to more than one structure, this method returns {@code null}.
   * Also, if the user has no access to the Structure plugin, returns null.</p>
   * 
   * <p>Note that the resulting structure may be archived.</p>
   * 
   * <p>Note that this method ignores {@link StructureAuth#isSecurityOverridden()}, unlike most other methods.</p>
   * 
   * @return the ID of the structure that the current user can see, only if it's the only one; {@code null} otherwise
   */
  @Nullable
  Long getSingleViewableStructureId();

  /**
   * <p>Checks if a specific issue (not row!) belongs to a specific structure.</p>
   * 
   * <p>Only issues that have been manually added to the structure are checked. 
   * Issues that have been added or removed by generators are not checked.</p>
   * 
   * <p>This method does not perform any permission checks.</p>
   *
   * @param issueId the ID of the issue. Nullable for convenience - if null, returns false
   * @param structureId the ID of the structure. Nullable for convenience - if null, returns false
   * @return true if the the issue is in the structure
   */
  boolean isIssueInStructureNoAccessCheck(@Nullable Long issueId, @Nullable Long structureId);

  /**
   * <p>Retrieves all structures that contain the specified issue. This is functionally the same as checking every forest
   * manually, but much faster.</p>
   * 
   * <p>Only issues that have been manually added to the structure are checked. 
   * Issues that have been added or removed by generators are not checked.</p>
   * 
   * <p>Note that forest changes may happen asynchronously, so consistency for the returned list is not guaranteed.
   * All we can say is that each structure in the returned list either contains the specified issue or has contained
   * it recently.</p>
   *
   * <p>This method returns only the structures that the current user can see.
   * If the user has no access to Structure plugin, returns empty list.
   * If {@link StructureAuth#isSecurityOverridden()} is true, these checks are not performed.</p>

   * <p>The resulting list is sorted by name, using the current user's locale.</p>
   *
   * @param issueId the ID of the issue. Nullable for convenience - if <code>null</code>, returns empty list
   * @return a list of structures, visible to the current user, containing the specified issue, not null
   */
  @NotNull
  List<Structure> getViewableStructuresWithIssue(@Nullable Long issueId);

  /**
   * <p>Calculates access level to a structure for the current user. 
   * If the user is not allowed to use Structure plugin, the result will always be {@link PermissionLevel#NONE}.</p>
   * 
   * <p>Note that this method ignores {@link StructureAuth#isSecurityOverridden()}, unlike most other methods.</p>
   *
   * @param structureId the ID of the structure checked. Nullable for convenience - if {@code null}, returns {@code NONE} 
   * @return the calculated access level, not null
   */
  @NotNull
  PermissionLevel getStructurePermission(@Nullable Long structureId);

  /**
   * <p>Calculates access level to a structure for the specified user.
   * If the user is not allowed to use Structure plugin, the result will always be {@link PermissionLevel#NONE}.</p>
   * 
   * <p>Note that {@link StructureAuth#isSecurityOverridden()} is not applicable here.</p>
   *
   * @param structureId the ID of the structure checked. Nullable for convenience - if {@code null}, returns {@code NONE}
   * @param user the user to check permissions for, {@code null} for anonymous
   * @return the calculated access level, not null
   */
  @NotNull
  PermissionLevel getStructurePermission(@Nullable Long structureId, @Nullable ApplicationUser user);

  /**
   * <p>Checks that the specified structure exists and the current user has access to it 
   * with the required permission level.</p>
   * 
   * <p>If structureId is null, or structure with that ID does not exist, false is returned.</p>
   * 
   * <p>If the user has no access to Structure plugin, returns false 
   * (unless {@link StructureAuth#isSecurityOverridden()} is true).</p>
   *
   * @param structureId the ID of the structure. Nullable for convenience - if {@code null}, returns {@code false}
   * @param requiredLevel when checking for user's permission, require that the user 
   *   has at least the specified permission level for the structure.
   *   Passing null or {@link PermissionLevel#NONE} effectively disables the structure permissions check, but we still
   *   check that the user has access to Structure plugin (unless {@link StructureAuth#isSecurityOverridden()} is true).
   * @return true if the specified structure exists and the user has access to it with the specified access level
   */
  boolean isAccessible(@Nullable Long structureId, @Nullable PermissionLevel requiredLevel);

  /**
   * <p>Checks that the specified user has the
   * {@link com.almworks.jira.structure.api.permissions.CoreAppPermissions#CONFIGURE_GENERATORS CONFIGURE_GENERATORS}
   * app permission and has at least {@link PermissionLevel#AUTOMATE AUTOMATE}
   * access level to the specified structure.</p>
   *
   * <p>Note that {@link StructureAuth#isSecurityOverridden()} is not applicable here.</p>
   *
   * @param structureId the ID of the structure checked. Nullable for convenience - if {@code null}, returns {@code false}
   * @param user the user to check permissions for, {@code null} for anonymous
   * @return true if the specified user is able to configure generators for the specified structure
   */
  boolean isGeneratorConfigurationAllowed(@Nullable Long structureId, @Nullable ApplicationUser user);

  /**
   * <p>Checks that the specified user has the
   * {@link com.almworks.jira.structure.api.permissions.CoreAppPermissions#CONFIGURE_EFFECTORS CONFIGURE_EFFECTORS}
   * app permission and has at least {@link PermissionLevel#AUTOMATE AUTOMATE}
   * access level to the specified structure.</p>
   *
   * <p>Note that {@link StructureAuth#isSecurityOverridden()} is not applicable here.</p>
   *
   * @param structureId the ID of the structure checked. Nullable for convenience - if {@code null}, returns {@code false}
   * @param user the user to check permissions for, {@code null} for anonymous
   * @return true if the specified user is able to configure effectors for the specified structure
   */
  boolean isEffectorConfigurationAllowed(@Nullable Long structureId, @Nullable ApplicationUser user);

  /**
   * <p>Checks that the specified user has the
   * {@link com.almworks.jira.structure.api.permissions.CoreAppPermissions#EXECUTE_EFFECTORS EXECUTE_EFFECTORS}
   * app permission and has at least {@link PermissionLevel#VIEW VIEW}
   * access level to the specified structure.</p>
   *
   * <p>Note that {@link StructureAuth#isSecurityOverridden()} is not applicable here.</p>
   *
   * @param structureId the ID of the structure checked. Nullable for convenience - if {@code null}, returns {@code false}
   * @param user the user to check permissions for, {@code null} for anonymous
   * @return true if the specified user is able to execute effectors for the specified structure
   */
  boolean isEffectorExecutionAllowed(@Nullable Long structureId, @Nullable ApplicationUser user);

  /**
   * <p>Creates an empty new structure. The structure returned is not yet persisted - you need to call the required 
   * setter methods and then call {@link Structure#saveChanges()} to write the new structure to the database 
   * and be able to fill it with items.</p>
   * 
   * <p>The structure will have the current user as its owner. 
   * This method checks that the current user can use Structure plugin and can create structures, unless
   * {@link StructureAuth#isSecurityOverridden()} is true.</p>
   * 
   * <p>Note that in order to write the contents of the structure you need to obtain an instance of 
   * {@link ForestSource} from {@link ForestService}:</p>
   * 
   * <pre>
   *   StructureManager sm = ...
   *   ForestService fs = ...
   *   Structure s = sm.createStructure().setName("new structure").saveChanges();
   *   ForestSpec spec = ForestSpec.structure(s.getId());
   *   ForestSource ufs = fs.getForestSource(spec);
   *   ufs.apply(new Add(CoreIdentities.issue(issueId), 0, 0, 0));
   * </pre>
   * */
  @NotNull
  Structure createStructure();

  /**
   * <p>Copies the structure and the forest it contains into a new structure.</p>
   *
   * <p>This method checks that the current user has all required permissions: 
   * to use Structure plugin, to create structures, and to {@link PermissionLevel#VIEW view} the copied structure; 
   * also, requires {@link PermissionLevel#ADMIN} access to copy permissions. 
   * All these checks are not performed if {@link StructureAuth#isSecurityOverridden()} is true.</p>
   *
   * @param structureId the ID of the structure. Nullable for convenience - if {@code null}, throws exception
   * @param newOwner the user who will be the owner of the new structure, if null - keep the old owner
   * @param copyPermissions if true, the permission list will be copied - this requires ADMIN permission level to the
   * original structure, if the user does not have it, StructureException is thrown
   * @return the freshly created structure. You can use {@link Structure#getId()} to learn its ID
   * @throws StructureException if the copied structure doesn't exist, the user has not enough permissions, or any other problem happens
   */
  @NotNull
  Structure copyStructure(@Nullable Long structureId, @Nullable ApplicationUser newOwner, boolean copyPermissions) throws StructureException;

  /**
   * <p>Copies the structure into a new structure - only the properties (such as name, description, permissions) 
   * are copied, not the forest.</p>
   *
   * <p>This method checks that the current user has all required permissions: 
   * to use Structure plugin, to create structures, and to {@link PermissionLevel#VIEW view} the copied structure; 
   * also, requires {@link PermissionLevel#ADMIN} access to copy permissions. 
   * All these checks are not performed if {@link StructureAuth#isSecurityOverridden()} is true.</p>
   *
   * @param structureId the ID of the structure. Nullable for convenience - if {@code null}, throws exception
   * @param newOwner the user who will be the owner of the new structure, if null - keep the old owner
   * @param copyPermissions if true, the permission list will be copied - this requires ADMIN permission level to the
   *   original structure, if the user does not have it, StructureException is thrown
   * @return the freshly created structure. You can use {@link Structure#getId()} to learn its ID
   * @throws StructureException if the copied structure doesn't exist, the user has not enough permissions, or any other problem happens
   */
  @NotNull
  Structure copyStructureWithoutForest(@Nullable Long structureId, @Nullable ApplicationUser newOwner, boolean copyPermissions) throws StructureException;

  /**
   * <p>Deletes the specified structure and its content.</p>
   *
   * <p>JIRA items such as issues or projects are not themselves deleted. Structure-owned items such as folders or
   * generators are deleted.</p>
   *
   * <p>This method checks that the current user can use Structure plugin and has
   * {@link PermissionLevel#ADMIN} access to the specified structure.
   * These checks are waived if {@link StructureAuth#isSecurityOverridden()} is true.</p>
   *
   * <p>This method will dispatch the {@link com.almworks.jira.structure.api.event.StructureDeletedEvent} after
   * structure deleting. An event listener can be registered using JIRA notification system.</p>
   *
   * @param id the ID of the structure. Nullable for convenience - if {@code null}, throws exception
   * @throws StructureException if the structure doesn't exist, is not accessible at ADMIN permission level, or there's problem deleting structure
   */
  void deleteStructure(@Nullable Long id) throws StructureException;

  /**
   * <p>Adds a structure listener. Whenever structure's <em>static content</em> is changed, the listeners are called
   * synchronously after the changes are persisted.</p>
   *
   * <p>See {@link StructureListener} for details about listener contract.</p>
   *
   * @param listener the listener
   * @see StructureListener
   */
  void addListener(@NotNull StructureListener listener);

  /**
   * <p>Removes the listener previously added via {@link #addListener(StructureListener)}.</p>
   *
   * @param listener listener to be removed
   */
  void removeListener(@NotNull StructureListener listener);
}
