package com.almworks.jira.structure.api.attribute.subscription;

import com.almworks.jira.structure.api.error.StructureException;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>{@code AttributeSubscriptionService} manages attribute subscriptions. A subscription allows the client to "watch" a particular
 * set of values, identified by a set of rows and attributes, retrieve updates to this set of values in versioned manner, and perform
 * background loading.</p>
 *
 * <p>Subscriptions are kept in a cache and may be deleted / closed if not used or if there are too many subscriptions.</p>
 *
 * @see AttributeSubscription
 */
@PublicApi
public interface AttributeSubscriptionService {
  /**
   * <p>Creates a new empty subscription.</p>
   *
   * <p>The subscription will be attached to the current user, and only that user will have the access to this subscription.</p>
   *
   * @return a new subscription
   * @see AttributeSubscription#changeSubscription
   */
  @NotNull
  AttributeSubscription createSubscription();

  /**
   * Retrieves a previously created subscription.
   * The ID of the subscription should come from a previous call to {@link AttributeSubscription#getSubscriptionId()}.
   *
   * @param subscriptionId subscription ID
   * @return the subscription
   * @throws StructureException if the current user is not the owner of the specified subscription, or if the subscription with this ID is not found
   */
  @NotNull
  AttributeSubscription getSubscription(@Nullable Long subscriptionId) throws StructureException;

  /**
   * Deletes the subscription. Any background loading is cancelled.
   *
   * @param subscriptionId subscription ID
   * @throws StructureException if the current user is not the owner of the specified subscription, or if the subscription with this ID is not found
   */
  void dropSubscription(@Nullable Long subscriptionId) throws StructureException;
}
