package com.almworks.jira.structure.api.settings;

import org.jetbrains.annotations.Nullable;

/**
 * <code>AutoSwitchStrategy</code> enum defines how the initial structure is selected when the user
 * opens issue page or project page.
 *
 * @see UISettings
 * @see StructureConfiguration#getUISettings
 * @author Igor Sereda
 */
public enum AutoSwitchStrategy {
  /**
   * When auto-switch is turned off, the "current" structure is opened on the page. Current structure is usually
   * the last structure that the user has viewed.
   */
  AUTOSWITCH_OFF("off"),

  /**
   * <p>When auto-switch is set to <code>STRUCTURE_WITH_ISSUE</code>, the structure displayed on the issue page
   * will be the structure that contains that issue.</p>
   *
   * <p>If the issue belongs to several structures (viewable by the user), or if it does not belong to any, the
   * behavior is not contractually defined. The implementation may make "best guess", based on the "current"
   * structure (the last structure the user has viewed) and the default structure for the project.</p>
   *
   * <p>This value does not make sense for the pages other than {@link StructurePage#ISSUE_VIEW}.</p>
   */
  STRUCTURE_WITH_ISSUE("withIssue"),

  /**
   * When auto-switch is set to <code>DEFAULT_STRUCTURE</code>, the default structure is always displayed when the
   * issue or project page is opened. The default structure for the viewed project (or the project of the viewed issue)
   * will be used.
   *
   * @see StructureConfiguration#getDefaultStructureId
   */
  DEFAULT_STRUCTURE("default"),
  ;

  /**
   * Defines system default auto-switch strategy for an issue page.
   */
  public static final AutoSwitchStrategy PLUGIN_DEFAULT_AUTOSWITCH_STRATEGY = STRUCTURE_WITH_ISSUE;
  
  private final String myStringCode;

  private AutoSwitchStrategy(String stringCode) {
    myStringCode = stringCode;
  }

  /**
   * @return a string that identifies this strategy (for serialization)
   */
  public String getStringCode() {
    return myStringCode;
  }

  /**
   * @return an auto-switch strategy with the matching code, or <code>null</code> if none exists (for serialization)
   */
  @Nullable
  public static AutoSwitchStrategy fromStringCode(String code) {
    if (code == null) return null;
    for (AutoSwitchStrategy strategy : AutoSwitchStrategy.values()) {
      if (code.equals(strategy.getStringCode())) return strategy;
    }
    return null;
  }
}
