package com.almworks.jira.structure.api.effector;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.atlassian.annotations.PublicSpi;
import com.atlassian.jira.util.ErrorCollection;
import org.jetbrains.annotations.NotNull;

import java.util.Map;

/**
 * <p>An effector is a component that generates effect descriptions based on a
 * forest, a set of attribute values, and a set of user-provided,
 * implementation-specific parameters.
 *
 * <p>The effector itself doesn't change anything, but only produces {@link
 * com.almworks.jira.structure.api.effect.StoredEffect effect
 * descirptions} for {@link com.almworks.jira.structure.api.effect.EffectProvider
 * effect providers}.
 */
@PublicSpi
public interface Effector {
  /**
   * <p>Returns an {@link EffectorFunction}, given the parameters. The returned
   * function will be called by Structure to produce effect descriptions.
   *
   * <p>For attributes, the effector must not use the
   * {@link com.almworks.jira.structure.api.attribute.StructureAttributeService
   * attribute service} directly, but require the attributes by calling
   * {@link EffectorContext#requireAttribute(AttributeSpec)}.
   *
   * <p>If it's impossible to create an effector function, e.g. because of
   * incorrect parameter values, the effector must return an {@link EffectorFunctionResponse#error(String, Object...)
   * error} response explaining the problem.
   *
   * @param parameters effector-specific parameters
   * @param context the context object
   * @return a {@link EffectorFunctionResponse response} containing either an
   * {@link EffectorFunction effector function} or a collection of errors
   */
  @NotNull
  EffectorFunctionResponse getEffectorFunction(
    @NotNull Map<String, Object> parameters, @NotNull EffectorContext context);

  /**
   * <p>Checks if this effector is currently available.
   *
   * <p>For example, an effector can be unavailable when a certain plug-in
   * is missing, or a certain application feature is disabled.
   *
   * <p>Effector is available by default.
   */
  default boolean isAvailable() {
    return true;
  }

  /**
   * Fills the parameter map for the edit form Velocity template with default
   * values for a new effector instance of this type.
   *
   * @param form form template parameters (output)
   * @throws EffectorUnavailableException if the effector is not available,
   * see {@link #isAvailable()}
   */
  void addDefaultFormParameters(@NotNull Map<String, Object> form) throws EffectorUnavailableException;

  /**
   * Given the effector-specific parameters, fills the parameter map for
   * the edit form Velocity template.
   *
   * @param parameters effector parameters (input)
   * @param form edit form template parameters (output)
   * @throws EffectorUnavailableException if the effector is not available,
   * see {@link #isAvailable()}
   */
  void addParametersToForm(@NotNull Map<String, Object> parameters, @NotNull Map<String, Object> form) throws EffectorUnavailableException;

  /**
   * <p>Validates the parameters from the edit HTML form and constructs a
   * serializable parameter map for {@link #getEffectorFunction(Map, EffectorContext)}.
   *
   * <p>Any errors for invalid input parameters must be reported to the provided
   * {@link ErrorCollection}; if it contains any error messages after this
   * method is called, they are reported to the user and the resulting map
   * is ignored.
   *
   * <p>The resulting parameter map could be stored in the database or
   * transferred over the wire, so it must be serializable to JSON. We advise
   * that you put in only simple objects (strings, numbers, booleans),
   * serializable maps, or lists thereof.
   *
   * @param form the parameters from the edit HTML form (input)
   * @param errors the collector for error messages (output)
   * @return serializable parameter map for {@link #getEffectorFunction(Map, EffectorContext)}
   * @throws EffectorUnavailableException if the effector is not available,
   * see {@link #isAvailable()}
   */
  @NotNull
  Map<String, Object> buildParametersFromForm(@NotNull Map<String, Object> form, @NotNull ErrorCollection errors) throws EffectorUnavailableException;

  /**
   * <p>Given the effector-specific parameters, fills the parameter map for
   * the summary Velocity template.
   *
   * <p>You have to override this method if your effector supports name
   * generation, i.e. has a Velocity template with name = "summary" and a
   * client-side descriptor.
   *
   * @param parameters effector parameters (input)
   * @param summaryParameters summary template parameters (output)
   */
  default void addParametersForSummary(@NotNull Map<String, Object> parameters, @NotNull Map<String, Object> summaryParameters) {}

  /**
   * <p>This method fills parameters for summary Velocity template. Similarly to {@link #addParametersForSummary(Map, Map)}.
   * Method should place a placeholder (for example <code>{myEffectorParam}</code>) for each parameter
   * that will be replaced with real value from effector edit form (see {@link #addParametersToForm(Map, Map)})
   * by corresponding client-side descriptor.
   *
   * <p>Each placeholder declared in this method will be automatically highlighted by Structure CSS on client side.
   *
   * <p>You have to override this method if your effector supports name
   * generation, i.e. has a Velocity template with name = "summary" and a
   * client-side descriptor.
   *
   * @param templatePlaceholders summary template parameter templates
   */
  default void addPlaceholdersForSummaryTemplate(@NotNull Map<String, String> templatePlaceholders) {}
}
