package com.almworks.jira.structure.api.attribute.loader.reduce;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.attribute.loader.AggregateAttributeContext;
import com.almworks.jira.structure.api.attribute.loader.basic.AbstractAggregateLoader;
import com.almworks.jira.structure.api.item.CoreIdentities;
import com.almworks.jira.structure.api.row.StructureRow;
import org.jetbrains.annotations.NotNull;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * <p>
 * A base class for aggregate loaders that calculate values using only the specified part of subtree.
 * The part of subtree to use is defined by {@link ReductionStrategy}, which is specified by {@code type} parameter in
 * the attribute specification.
 * </p>
 *
 * <p>Here and in other classes, the process of combining the current value with values calculated for partial subtree
 * into one value is called <em>reduction</em>.</p>
 *
 * This class:
 * <ol>
 * <li>checks for errors,</li>
 * <li>filters out values for generator rows,</li>
 * <li>passes values to the reduction strategy.</li>
 * </ol>
 *
 * @param <T> type of reduced values
 * @see ValueReducer
 * @see ReductionStrategy
 */
public abstract class ReducingAggregateLoader<T> extends AbstractAggregateLoader<T> implements ValueReducer<T> {
  private final ReductionStrategy<T> myReductionStrategy;

  public ReducingAggregateLoader(AttributeSpec<T> spec) {
    super(spec);
    myReductionStrategy = ReductionStrategy.forAttributeSpec(spec);
  }

  @Override
  public AttributeValue<T> loadValue(List<AttributeValue<T>> childrenValues, AggregateAttributeContext context) {
    List<StructureRow> children = context.getChildren();
    if (children.stream().anyMatch(row -> CoreIdentities.isAutomation(row.getItemId()))) {
      assert children.size() == childrenValues.size();
      childrenValues = IntStream.range(0, children.size())
        .filter(idx -> !CoreIdentities.isAutomation(children.get(idx).getItemId()))
        .mapToObj(childrenValues::get)
        .collect(Collectors.toList());
    }
    AttributeValue<T> error = firstChildError(childrenValues);
    if (error != null) return error;
    return myReductionStrategy.apply(() -> getSelfValue(context), childrenValues, this);
  }

  @NotNull
  protected abstract AttributeValue<T> getSelfValue(AggregateAttributeContext context);
}
