package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.integers.LongSet;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;

/**
 * <p>Common base interface for all row-based attribute loaders.</p>
 *
 * <p>All row-based attribute loaders provide an attribute value based on a row in a forest. The values are stored in a per-forest
 * value cache.</p>
 *
 * @param <T> type of the loaded value
 *
 * @see AttributeLoader
 * @see SingleRowAttributeLoader
 * @see AggregateAttributeLoader
 * @see PropagateAttributeLoader
 * @see ScanningAttributeLoader
 */
@PublicSpi
public interface RowAttributeLoader<T> extends AttributeLoader<T> {
  /**
   * <p>Optional method to perform any bulk actions on a set of rows before the loading function is called for each row.</p>
   *
   * <p>The results of the preloading should be stored using {@link AttributeContext#putObject}.
   * <strong>Do not store the preloading results as loader's instance fields!</strong> All loaders should be stateless.</p>
   *
   * <p>Note that this method may be called several times during a single loading operation, for arbitrary sets of rows, including those
   * that were already pre-loaded. Be careful not to overwrite one execution's result with another's.</p>
   *
   * <p>The loader's dependencies may not be loaded at the time of the call. In fact, there's no way to access the dependency values.</p>
   *
   * <p>Note that {@code rowIds} will never contain the super-root row id ({@code -1}), even if the value for it is about to be loaded.
   * See {@link com.almworks.jira.structure.api.row.SuperRootRow} for details about the super-root.</p>
   *
   * @param rowIds set of row IDs for which attribute values are needed
   * @param forest forest for which attribute values are being loaded
   * @param context loading context
   */
  default void preload(@NotNull LongSet rowIds, @NotNull ItemForest forest, @NotNull AttributeContext context) {}

  /**
   * <p>Allows the loader to declare dependency on the whole forest. In that case, the value loaded might depend on the forest and dependencies for
   * the current row. (No dependencies on other values.)</p>
   *
   * <p>The effect of such declaration is that the values of this attribute, as well as any other values depending on them, will be invalidated
   * as soon as there's any forest or item changes.</p>
   *
   * <p>You should almost never override this method. Introducing a whole-forest attribute takes toll on application
   * performance.</p>
   *
   * @return true if the attribute is whole forest-dependent
   */
  default boolean isWholeForestDependent() { return false; }
}
