package com.almworks.jira.structure.api.util;

import com.almworks.jira.structure.api.attribute.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

import static com.almworks.jira.structure.api.util.JsonMapUtil.unwrapJsonCollection;

/**
 * Usable by AttributeSpec and other Specs that have Map parameters.
 */
public class SpecParams extends MapObject {
  private static final Logger logger = LoggerFactory.getLogger(SpecParams.class);

  public SpecParams(Map<String, Object> map) {
    super(map);
  }

  @Nullable
  @Override
  public SpecParams getObject(String name) {
    Object value = myObject.opt(name);
    value = unwrapJsonCollection(value);
    if (!(value instanceof Map)) {
      return null;
    }
    //noinspection unchecked
    return new SpecParams((Map<String, Object>) value);
  }

  @Nullable
  public <T> AttributeSpec<T> getAttributeParameter(ValueFormat<T> expectedFormat) {
    return getAttributeParameter(SharedAttributeSpecs.Param.ATTRIBUTE, expectedFormat);
  }

  @Nullable
  public <T> AttributeSpec<T> getAttributeParameter(String name, ValueFormat<T> expectedFormat) {
    SpecParams attributeObject = getObject(name);
    if (attributeObject == null) {
      return null;
    }
    String format = attributeObject.getString(SharedAttributeSpecs.Param.FORMAT);
    if (format != null && expectedFormat != null){
      if (!expectedFormat.getFormatId().equals(format) && !ValueFormat.ANY.getFormatId().equals(format)) {
        return null;
      }
    }
    if (expectedFormat == null) {
      ValueFormat<?> standardFormat = ValueFormat.getStandardFormat(format);
      if (standardFormat != null) {
        expectedFormat = (ValueFormat<T>) standardFormat;
      }
    }
    return buildAttributeFromParamsMap(attributeObject, expectedFormat);
  }

  @Nullable
  private <T> AttributeSpec<T> buildAttributeFromParamsMap(@NotNull SpecParams attributeObject, ValueFormat<T> expectedFormat) {
    if (expectedFormat == null) return null;
    String id = attributeObject.getString(SharedAttributeSpecs.Param.ID);
    Object params = attributeObject.get(SharedAttributeSpecs.Param.PARAMS);
    params = unwrapJsonCollection(params);
    //noinspection unchecked
    Map<String, Object> paramsMap = params instanceof Map ? (Map<String, Object>) params : null;
    try {
      return new AttributeSpec<T>(id, expectedFormat, paramsMap);
    } catch (IllegalArgumentException e) {
      logger.debug("invalid params - cannot extract AttributeSpec: " + attributeObject);
      return null;
    }
  }

}
