package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.row.StructureRow;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>A single row attribute loader provides an attribute value based on a single forest row. The value must not depend on where this row
 * is in the forest.</p>
 *
 * @param <T> type of the loaded value
 *
 * @see AttributeLoader
 */
@PublicSpi
public interface SingleRowAttributeLoader<T> extends RowAttributeLoader<T> {
  /**
   * <p>The loading function. The implementation is expected to take {@code row} and provide a value.</p>
   *
   * <p>Before this method is called, {@link #preload} has been called for a set of rows that included this row, and that was
   * a part of the same loading process, sharing the same context (so all values put into the context are still there).</p>
   *
   * <p>The exception to this rule is the super-root row, which is not included in the preload call. See {@link com.almworks.jira.structure.api.row.SuperRootRow} for more information
   * about the super-root.</p>
   *
   * <p>If the value is missing or empty, the loader should return {@link AttributeValue#undefined()}. If for some reason this loader
   * is not applicable for the given row, it should return {@code null}.</p>
   *
   * @param row the row to load the value for
   * @param context loading context
   * @return attribute value, or {@code null} to pass on loading a value for this item
   */
  @Nullable
  AttributeValue<T> loadValue(@NotNull StructureRow row, @NotNull SingleRowAttributeContext context);
}
