package com.almworks.jira.structure.api;

import com.almworks.jira.structure.api.attribute.StructureAttributeService;
import com.almworks.jira.structure.api.attribute.subscription.AttributeSubscriptionService;
import com.almworks.jira.structure.api.backup.StructureBackupManager;
import com.almworks.jira.structure.api.event.IssueEventBridge;
import com.almworks.jira.structure.api.folder.FolderManager;
import com.almworks.jira.structure.api.forest.*;
import com.almworks.jira.structure.api.generator.GeneratorManager;
import com.almworks.jira.structure.api.item.*;
import com.almworks.jira.structure.api.job.ScheduledJobManager;
import com.almworks.jira.structure.api.job.StructureJobManager;
import com.almworks.jira.structure.api.process.ProcessHandleManager;
import com.almworks.jira.structure.api.property.StructurePropertyService;
import com.almworks.jira.structure.api.query.StructureQueryBuilderFactory;
import com.almworks.jira.structure.api.query.StructureQueryParser;
import com.almworks.jira.structure.api.row.RowManager;
import com.almworks.jira.structure.api.settings.StructureConfiguration;
import com.almworks.jira.structure.api.structure.StructureManager;
import com.almworks.jira.structure.api.structure.favorite.StructureFavoriteManager;
import com.almworks.jira.structure.api.sync.StructureSyncManager;
import com.almworks.jira.structure.api.sync.SyncAuditLog;
import com.almworks.jira.structure.api.view.StructureViewManager;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

/**
 * <p>
 * StructureComponents is a convenience interface that can be used to retrieve all public services
 * of the Structure plugin (except internal and experimental APIs).
 * You can also get those services from ComponentAccessor or by declaring them
 * in a constructor of your component and have them injected.
 * </p>
 * <p>
 * The convenience is in reducing the number of parameters you declare in the constructor of your
 * component.
 * </p>
 * <p>
 * Note that this service returns the original instances of those services, unlike when the proxies for the
 * services are injected.
 * </p>
 * <p>
 * This class also provides some frequently-used Atlassian interfaces.
 * </p>
 *
 * @author Igor Sereda
 */
@PublicApi
public interface StructureComponents {
  /**
   * Returns the component that's used to manage structures.
   */
  @NotNull
  StructureManager getStructureManager();

  /**
   * Returns a service that lets you retrieve {@link ForestSource} for a structure or a given {@link ForestSpec}.
   */
  @NotNull
  ForestService getForestService();

  /**
   * Returns the service that is used to calculate and get field values and other attributes for items.
   */
  @NotNull
  StructureAttributeService getAttributeService();

  /**
   * Returns the service that is used to manage subscriptions to attributes. It's an async way to retrieve attributes.
   */
  @NotNull
  AttributeSubscriptionService getAttributeSubscriptionService();

  /**
   * Returns the component that's used to manage Structure add-on configuration.
   */
  @NotNull
  StructureConfiguration getStructureConfiguration();

  /**
   * Returns the component that's used to create and update synchronizers.
   */
  @NotNull
  StructureSyncManager getSyncManager();

  /**
   * Returns the component that's used to back up and restore Structure data.
   */
  @NotNull
  StructureBackupManager getBackupManager();

  /**
   * Returns the component that's used to run asynchronous jobs.
   */
  @NotNull
  StructureJobManager getJobManager();

  /**
   * Returns the component that is used to schedule recurring asynchronous jobs.
   */
  @NotNull
  ScheduledJobManager getScheduledJobManager();

  /**
   * Returns the component that's used to listen and report issue events.
   */
  @NotNull
  IssueEventBridge getIssueEventBridge();

  /**
   * Returns the component that's used to work with Structure views.
   */
  @NotNull
  StructureViewManager getViewManager();

  /**
   * Returns the component that's used to access and manage "favorite" relations between users and structures.
   */
  @NotNull
  StructureFavoriteManager getFavoriteManager();

  /**
   * Returns the component that is used to create Structure queries by parsing them from a String.
   */
  @NotNull
  StructureQueryParser getStructureQueryParser();

  /**
   * Returns the component that is used to create new instances of Structure query builders.
   */
  @NotNull
  StructureQueryBuilderFactory getStructureQueryBuilderFactory();

  /**
   * Returns component that tracks long-running background processes and shows progress to the user.
   */
  @NotNull
  ProcessHandleManager getProcessHandleManager();

  /**
   * Returns row manager, which is responsible for managing a mapping between numeric row IDs and item IDs.
   */
  @NotNull
  RowManager getRowManager();

  /**
   * Returns the component that synchronizers use to store audit information about their execution and actions
   * they've taken.
   */
  @NotNull
  SyncAuditLog getSyncAuditLog();

  /**
   * Returns the service responsible for storing and retrieving generator instances.
   */
  @NotNull
  GeneratorManager getGeneratorManager();

  /**
   * Returns a service that resolves {@link ItemIdentity} into an object that represents that item.
   */
  @NotNull
  ItemResolver getItemResolver();

  /**
   * Returns a service responsible for storing and retrieving folders.
   */
  @NotNull
  FolderManager getFolderManager();

  /**
   * Returns a service that lets you associate arbitrary properties with a structure.
   */
  @NotNull
  StructurePropertyService getStructurePropertyService();

  /**
   * Returns a service that tracks item updates throughout the system and also allows reporting updates.
   */
  @NotNull
  ItemTracker getItemTracker();
}
