package com.almworks.jira.structure.api.structure;

import com.almworks.integers.LongObjMap;
import com.almworks.jira.structure.api.forest.ForestSource;
import com.almworks.jira.structure.api.forest.action.ForestAction;
import com.almworks.jira.structure.api.job.StructureJobManager;
import com.almworks.jira.structure.api.structure.history.HistoryEntry;
import com.atlassian.annotations.PublicSpi;

import java.util.Map;

/**
 * <p><code>StructureListener</code> allows you to listen to changes to the <em>static</em> content in structures.</p>
 *
 * <p>Each structure is associated with an {@link ForestSource}, which represents its contents visible to the user.
 *  The contents consist of <em>static</em> and <em>dynamic</em> rows. 
 *  Dynamic rows are all rows inserted or removed by generators. 
 *  Static rows are the rest: the generators themselves and the items that were added manually or by synchronizers.
 *  A successful update made via {@link ForestSource#apply(ForestAction, Map)} may result in changes
 *  to either static content, dynamic content, neither (if it doesn't change the contents) or both. 
 *  A <code>StructureListener</code> gets notified only of changes to static content.</p>
 *  
 * <p>This interface is closely related to the structure history: basically, you'll get notified about new history entries.
 *  The notification can be batched if one logical update results in several history entries.</p>
 *  
 * <h2>Examples:</h2>
 *  
 * <ol>
 *   <li>Any change to the structure that doesn't contain generators will be reported.</li>
 *   <li>If you move an issue from one group to another in a structure with Grouper generator, this might only result
 *   in a change to an issue - a change to the dynamic content. This change is not reported.</li>
 *   <li>If you add an issue in the domain of the Structure inserter, this may constitute both a change to dynamic content 
 *   (with regard to the structure where the change is made) and a change to static content (with regard to the inserted structure).
 *   Only the latter will be reported.</li>
 * </ol>
 *  
 * <p>The listener is called after changes has been persisted.</p>
 *
 * <p>JIRA Data Center note: only changes happened on the local node are reported.</p>
 * 
 * <p>The listener is called <strong>in the same thread</strong> that has made the change. When the listener
 *  is called <strong>no global lock is held</strong> by Structure. The listener implementation should be as fast
 *  as reasonably possible and not perform any blocking activity (take locks, wait, do I/O).</p>
 *  
 * <p>The <strong>order of calls is not guaranteed</strong>, that is, if there are two changes - A, and then B, your
 *  listener may be notified about B first, and then about A.</p>
 *
 * @see StructureManager#addListener(StructureListener)
 *
 * @author Igor Sereda
 */
@PublicSpi
public interface StructureListener {
  /**
   * <p>Called after a related group of changes has been written to the database.</p>
   * <p>
   * This method is called in the same thread where the changes were made, after the changes were made.
   * If you need to perform blocking activity (take locks, wait, do I/O), it is recommended to offload these
   * tasks to another thread (for example, via {@link StructureJobManager}).
   * 
   * @param structureChanges a map from structure ID into the {@link HistoryEntry} describing the changes
   * */
  void onStructuresChanged(LongObjMap<HistoryEntry> structureChanges);
}
