package com.almworks.jira.structure.api.permissions;

import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableSet;

/**
 * Core app-level permissions checked by Structure.
 * @see com.almworks.jira.structure.api.settings.StructureConfiguration
 */
@PublicApi
public interface CoreAppPermissions {
  /**
   * Permission to use Structure.
   */
  StructureAppPermission USE = new StructureAppPermission("use");

  /**
   * Permission to create new structures.
   *
   * The user must also have the {@link #USE} permission.
   */
  StructureAppPermission CREATE_STRUCTURE = new StructureAppPermission("createStructure");

  /**
   * Permission to configure and control synchronizers.
   *
   * The user must also have the {@link #USE} permission and
   * {@link PermissionLevel#ADMIN} access to the structure.
   */
  StructureAppPermission SYNCHRONIZATION = new StructureAppPermission("synchronization");

  /**
   * Permission to use automation (generators and effectors).
   *
   * The user must also have the {@link #USE} permission and
   * {@link PermissionLevel#AUTOMATE} access to the structure.
   */
  StructureAppPermission AUTOMATION = new StructureAppPermission("automation");

  /**
   * Permission to create and configure generators.
   *
   * The user must also have the {@link #USE} and {@link #AUTOMATION}
   * permissions and {@link PermissionLevel#AUTOMATE} access to the structure.
   */
  StructureAppPermission CONFIGURE_GENERATORS = new StructureAppPermission("configureGenerators");

  /**
   * Permission to create and configure effectors.
   *
   * The user must also have the {@link #USE} and {@link #AUTOMATION}
   * permissions and {@link PermissionLevel#AUTOMATE} access to the structure.
   */
  StructureAppPermission CONFIGURE_EFFECTORS = new StructureAppPermission("configureEffectors");

  /**
   * Permission to execute effectors installed in structures.
   *
   * The user must also have the {@link #USE} permission,
   * the {@link com.atlassian.jira.permission.GlobalPermissionKey#BULK_CHANGE}
   * Jira permission, and {@link PermissionLevel#VIEW} access to the structure.
   */
  StructureAppPermission EXECUTE_EFFECTORS = new StructureAppPermission("executeEffectors");

  /**
   * Permission to execute effectors on query results.
   *
   * The user must also have the {@link #USE}, {@link #AUTOMATION},
   * {@link #CONFIGURE_EFFECTORS}, and {@link #EXECUTE_EFFECTORS} permissions
   * and the {@link com.atlassian.jira.permission.GlobalPermissionKey#BULK_CHANGE}
   * Jira permission.
   */
  StructureAppPermission EXECUTE_EFFECTORS_ON_QUERIES = new StructureAppPermission("executeEffectorsOnQueries");

  /**
   * Permission to manage global saved columns: edit, remove, add new.
   *
   * The user must also have the {@link #USE} permission
   * and the {@link com.atlassian.jira.permission.GlobalPermissionKey#CREATE_SHARED_OBJECTS}
   * Jira permission.
   */
  StructureAppPermission MANAGE_GLOBAL_SAVED_COLUMNS = new StructureAppPermission("manageGlobalSavedColumns");

  Set<StructureAppPermission> ALL = unmodifiableSet(new LinkedHashSet<>(asList(
    USE, CREATE_STRUCTURE, SYNCHRONIZATION, AUTOMATION, CONFIGURE_GENERATORS,
    CONFIGURE_EFFECTORS, EXECUTE_EFFECTORS, EXECUTE_EFFECTORS_ON_QUERIES,
    MANAGE_GLOBAL_SAVED_COLUMNS
  )));

  @Nullable
  static StructureAppPermission byKey(@Nullable String key) {
    return ALL.stream().filter(p -> p.getKey().equals(key)).findAny().orElse(null);
  }
}
