package com.almworks.jira.structure.api.item.generic;

import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.util.*;
import com.atlassian.annotations.PublicApi;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * <p>Class responsible for storing data related to generic item.</p>
 *
 * <p>{@code GenericItem} objects may be obtained with {@link GenericItemService#getItem(ItemIdentity)} or with
 * {@link GenericItemManager#getItem(ItemIdentity)}.</p>
 *
 * @see GenericItemService
 * @see GenericItemManager
 */
@PublicApi
public final class GenericItem {
  @NotNull
  private final String myName;
  @Nullable
  private final Long myOwningStructure;
  @Nullable
  private final String myDescription;
  @NotNull
  private final Map<String, Object> myParameters;

  @NotNull
  public static GenericItem.Builder named(@NotNull String name) {
    return new GenericItem.Builder(name);
  }

  @NotNull
  public static Builder copy(@NotNull GenericItem genericItem) {
    return new Builder(genericItem.getName()).setOwningStructure(genericItem.getOwningStructure())
      .setDescription(genericItem.getDescription()).setParameters(genericItem.getParameters());
  }

  private GenericItem(@NotNull String name, @Nullable Long owningStructure, @Nullable String description,
    @Nullable Map<String, Object> parameters)
  {
    myName = StructureUtil.nonBlank(name);
    myOwningStructure = owningStructure;
    myDescription = description;
    myParameters = JsonMapUtil.copyParameters(parameters, false, true, false);
  }

  @NotNull
  public String getName() {
    return myName;
  }

  @Nullable
  public Long getOwningStructure() {
    return myOwningStructure;
  }

  @Nullable
  public String getDescription() {
    return myDescription;
  }

  @NotNull
  public Map<String, Object> getParameters() {
    return myParameters;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    GenericItem that = (GenericItem) o;
    return myName.equals(that.myName) && Objects.equals(myOwningStructure, that.myOwningStructure) && Objects.equals(myDescription,
      that.myDescription) && myParameters.equals(that.myParameters);
  }

  @Override
  public int hashCode() {
    return Objects.hash(myName, myOwningStructure, myDescription, myParameters);
  }

  @Override
  public String toString() {
    return "GenericItem{" +
      "myName='" + myName + '\'' +
      ", myOwningStructure=" + myOwningStructure +
      ", myDescription='" + myDescription + '\'' +
      ", myParameters=" + myParameters +
      '}';
  }

  @PublicApi
  public static final class Builder {
    @NotNull
    private String myName;
    @Nullable
    private Long myOwningStructure;
    @Nullable
    private String myDescription;
    @Nullable
    private Map<String, Object> myParameters;

    private Builder(@NotNull String name) {
      myName = checkName(name);
    }

    @NotNull
    public Builder setName(@NotNull String name) {
      myName = checkName(name);
      return this;
    }

    private String checkName(String name) {
      if (StringUtils.isBlank(name)) {
        throw new IllegalArgumentException("name cannot be empty");
      }
      return StringUtils.abbreviate(name, Limits.MAX_NAME_LENGTH);
    }

    @NotNull
    public Builder setOwningStructure(@Nullable Long owningStructure) {
      myOwningStructure = owningStructure;
      return this;
    }

    @NotNull
    public Builder setDescription(@Nullable String description) {
      myDescription = description;
      return this;
    }

    @NotNull
    public Builder setParameter(@NotNull String name, @Nullable Object value) {
      if (StringUtils.isBlank(name)) {
        throw new IllegalArgumentException("parameter name cannot be empty");
      }
      if (value == null) {
        if (myParameters != null) {
          myParameters.remove(name);
        }
      } else {
        if (myParameters == null) {
          myParameters = new LinkedHashMap<>();
        }
        myParameters.put(name, value);
      }
      return this;
    }

    @NotNull
    public Builder setParameters(@Nullable Map<String, Object> params) {
      myParameters = params != null ? new LinkedHashMap<>(params) : null;
      return this;
    }

    @NotNull
    public GenericItem build() {
      return new GenericItem(myName, myOwningStructure, myDescription, myParameters);
    }
  }
}