package com.almworks.jira.structure.api.forest.item;

import com.almworks.integers.LongArray;
import com.almworks.integers.LongIterator;
import com.almworks.jira.structure.api.forest.raw.ArrayForest;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.row.*;
import com.atlassian.annotations.PublicApi;
import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;

import javax.annotation.concurrent.Immutable;
import java.util.Collections;
import java.util.Map;

@PublicApi
@Immutable
public final class ImmutableItemForest implements ItemForest {
  public static final ItemForest EMPTY = new ImmutableItemForest(
    new ArrayForest().makeImmutable(), ImmutableMap.<Long, StructureRow>of());

  private final Forest myForest;
  private final Map<Long, StructureRow> myRows;

  private ImmutableItemForest(@NotNull Forest forest, @NotNull Map<Long, StructureRow> rows) {
    myForest = forest;
    myRows = rows;
  }

  public static ImmutableItemForest of(@NotNull Forest forest, @NotNull Map<Long, StructureRow> rows) {
    checkAllRowsPresent(forest, rows);
    forest = new ArrayForest(forest).makeImmutable();
    rows = ImmutableMap.copyOf(rows);
    return new ImmutableItemForest(forest, rows);
  }

  public static ImmutableItemForest of(@NotNull StructureRow row) {
    return new ImmutableItemForest(new ArrayForest(row.getRowId()), Collections.singletonMap(row.getRowId(), row));
  }

  private static void checkAllRowsPresent(@NotNull Forest forest, @NotNull Map<Long, ?> rows) throws IllegalArgumentException {
    LongArray missing = null;
    for (LongIterator it : forest.getRows()) {
      if (rows.get(it.value()) == null) {
        if (missing == null) {
          missing = new LongArray();
        }
        missing.add(it.value());
      }
    }
    if (missing != null) {
      throw new IllegalArgumentException("No data for " + missing);
    }
  }

  @NotNull
  public static ImmutableItemForest copy(@NotNull ItemForest copyFrom) {
    if (copyFrom instanceof ImmutableItemForest) {
      return (ImmutableItemForest) copyFrom;
    } else {
      ArrayForest forest = new ArrayForest(copyFrom.getForest()).makeImmutable();
      return rearrangeItemForest(copyFrom, forest);
    }
  }

  @NotNull
  public static ImmutableItemForest copySubtree(@NotNull ItemForest copyFrom, long rootId) {
    Forest subtree = copyFrom.getForest().subtree(rootId);
    return rearrangeItemForest(copyFrom, subtree);
  }

  /**
   * Returns an {@link ItemForest} based on {@code newForest}, with rows taken
   * from {@code source}. {@code newForest} cannot contain any new rows, it must
   * be a permutation or a sub-forest of {@code source.getForest()}.
   *
   * @throws MissingRowException if some of the rows in the newForest are not found
   */
  @NotNull
  public static ImmutableItemForest rearrangeItemForest(@NotNull ItemForest source, @NotNull Forest newForest) throws MissingRowException {
    ImmutableMap.Builder<Long, StructureRow> builder = ImmutableMap.builder();
    source.scanAllRows(newForest.getRows(), row -> builder.put(row.getRowId(), row));
    return new ImmutableItemForest(newForest, builder.build());
  }

  @NotNull
  @Override
  public Forest getForest() {
    return myForest;
  }

  @NotNull
  @Override
  public StructureRow getRow(long rowId, @NotNull ItemAccessMode access) throws MissingRowException {
    StructureRow row = myRows.get(rowId);
    if (row == null) {
      throw new MissingRowException(rowId);
    }
    if (access == ItemAccessMode.SKIP_ACCESS_CHECK) {
      return StructureRows.makeUnchecked(row);
    }
    return row;
  }
}
