package com.almworks.jira.structure.api.event;

import com.almworks.integers.LongList;
import com.atlassian.jira.event.issue.IssueEvent;
import com.atlassian.jira.issue.Issue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * {@code IssueChangeEvent} represents an event that has happened with a specific issue.
 * It may be constructed with an instance of {@link IssueEvent} received by issue listener.
 */
public class IssueChangeEvent extends JiraChangeEvent {
  private final long myIssueId;

  @Nullable
  private final IssueEvent myJiraEvent;

  public IssueChangeEvent(@NotNull JiraChangeType changeType, long issueId, @Nullable IssueEvent jiraEvent) {
    super(changeType);
    myIssueId = issueId;
    myJiraEvent = jiraEvent;
  }

  public IssueChangeEvent(@NotNull JiraChangeType changeType, long issueId) {
    this(changeType, issueId, null);
  }

  public IssueChangeEvent(@NotNull JiraChangeType changeType, @NotNull IssueEvent jiraEvent) {
    super(changeType);
    myJiraEvent = jiraEvent;
    long id = 0;
    if (jiraEvent != null) {
      Issue issue = jiraEvent.getIssue();
      if (issue != null) {
        Long issueId = issue.getId();
        if (issueId != null) {
          id = issueId;
        }
      }
    }
    assert id != 0 : changeType + " " + jiraEvent;
    myIssueId = id;
  }


  /**
   * @return changed issue ID
   */
  public long getIssueId() {
    return myIssueId;
  }

  /**
   * @return the instance of JIRA's {@link com.atlassian.jira.event.issue.IssueEvent} that caused this event, if any
   */
  @Nullable
  public IssueEvent getJiraEvent() {
    return myJiraEvent;
  }

  public String toString() {
    return getChangeType() + "(" + myIssueId + ')';
  }

  public LongList getAffectedIssuesSorted() {
    return new LongList.Single(myIssueId);
  }

  public static long getIssueId(JiraChangeEvent event) {
    if (event instanceof IssueChangeEvent) {
      return ((IssueChangeEvent) event).getIssueId();
    }
    return 0;
  }
}
