package com.almworks.jira.structure.api.export;

import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.Nullable;

import java.util.Date;

/**
 * <p>{@code ExportCell} represents a single cell in the underlying table-like medium (e.g. an Excel sheet or an HTML
 * table) that a structure can be exported into. Structure creates a new cell for each column and each issue in the
 * exported structure (or part of a structure), and passes an {@code ExportCell} instance to the {@link ExportRenderer}
 * responsible for the column for rendering the cell. Please note that the same {@code ExportCell} instances may be
 * reused in many calls to export renderers in order to put less stress on the garbage collector.</p>
 *
 * <p>An {@code ExportCell} has a bunch of methods to put different types of values into the cell. The actual
 * representation of a particular value type may depend on the underlying medium, the user's locale, and other settings.
 * If not specified otherwise, passing a {@code null} value to any of the value-setting methods clears the cell.
 * Media-specific sub-interfaces may add methods for setting other types of values and configuring the cell.</p>
 *
 * @see ExportColumn
 * @see ExportRenderer
 */
@PublicApi
public interface ExportCell {
  /**
   * Set a simple text string as the cell value.
   *
   * @param text The text.
   */
  void setText(@Nullable String text);

  /**
   * Set a number as the cell value. The number may be formatted according to the user's locale and other settings.
   *
   * @param number The number.
   */
  void setNumber(@Nullable Number number);

  /**
   * Set a percentage as the cell value. The percentage may be represented as text or graphically, depending on the
   * medium and other settings. This is the method used by Structure's Progress and Formula columns.
   *
   * @param percentage The percentage, 1.0 is interpreted as 100%, negative values are not supported
   */
  void setPercentage(@Nullable Double percentage);

  /**
   * Set a date as the cell value. The date may be formatted according to the user's locale and other settings.
   *
   * @param date The date.
   */
  void setDate(@Nullable Date date);

  /**
   * Set a date and time as the cell value. The value may be formatted according to the user's locale and other
   * settings.
   *
   * @param date The date and time.
   */
  void setDatetime(@Nullable Date date);

  /**
   * Set a duration as the cell value. The duration may be converted into a human-readable form according to the user's
   * locale and other settings. Examples of duration representations are "2h 30m", "2 hours, 30 minutes", or "2:30".
   *
   * @param duration The duration, in seconds.
   */
  void setDuration(@Nullable Long duration);

  /**
   * Set a hyperlink as the cell value.
   *
   * @param text The text of the hyperlink. If {@code null}, the URL is used as the text.
   * @param url The target URL of the hyperlink. If {@code null}, the cell is cleared.
   */
  void setHyperlink(@Nullable String text, @Nullable String url);
}
