package com.almworks.jira.structure.api.effector;

import com.almworks.jira.structure.api.util.I18nText;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import java.util.*;

/**
 * A response object returned by {@link Effector#getEffectorFunction(Map, EffectorContext)},
 * containing either an {@link EffectorFunction} or a collection of error
 * messages.
 */
@PublicApi
public abstract class EffectorFunctionResponse {
  private EffectorFunctionResponse() {}

  /**
   * Constructs and returns a valid response.
   *
   * @param effectorFunction the returned effector function
   * @return a valid response
   */
  public static EffectorFunctionResponse valid(@NotNull EffectorFunction effectorFunction) {
    return new Valid(effectorFunction);
  }

  /**
   * Constructs and returns an error response with a single error message.
   *
   * @param i18nKey i18n key for the error message
   * @param parameters parameters for the error message
   * @return an error response
   */
  public static EffectorFunctionResponse error(@NotNull String i18nKey, Object... parameters) {
    return new Errors(Collections.singletonList(new I18nText(i18nKey, parameters)));
  }

  /**
   * Constructs and returns an error response with one or more error messages.
   *
   * @param errors error messages
   * @return an error response
   * @throws IllegalArgumentException if no error messages are given
   */
  public static EffectorFunctionResponse errors(@NotNull I18nText... errors) {
    return new Errors(Arrays.asList(errors));
  }

  /**
   * Constructs and returns an error response with one or more error messages.
   *
   * @param errors error messages
   * @return an error response
   * @throws IllegalArgumentException if no error messages are given
   */
  public static EffectorFunctionResponse errors(@NotNull List<I18nText> errors) {
    return new Errors(errors);
  }


  /**
   * A valid effector response.
   */
  public static final class Valid extends EffectorFunctionResponse {
    @NotNull
    private final EffectorFunction myEffectorFunction;

    private Valid(@NotNull EffectorFunction effectorFunction) {
      myEffectorFunction = effectorFunction;
    }

    @NotNull
    public EffectorFunction getEffectorFunction() {
      return myEffectorFunction;
    }
  }


  /**
   * An invalid effector response with error messages.
   */
  public static final class Errors extends EffectorFunctionResponse {
    @NotNull
    private final List<I18nText> myErrors;

    private Errors(@NotNull List<I18nText> errors) {
      if (errors.size() == 0) throw new IllegalArgumentException("No errors in arguments");
      myErrors = errors;
    }

    @NotNull
    public List<I18nText> getErrors() {
      return myErrors;
    }
  }
}
