package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Contains information about problems experienced when loading a particular attribute.
 *
 * @see ValuesMeta
 */
@PublicApi
public class AttributeErrorInfo {
  private final AttributeSpec<?> myAttributeSpec;
  private final Long myRowId;
  private final ItemIdentity myItemId;
  private final Throwable myCause;

  public AttributeErrorInfo(@NotNull AttributeSpec<?> attributeSpec, @Nullable Long rowId,
    @Nullable ItemIdentity itemId, @Nullable Throwable cause)
  {
    myAttributeSpec = attributeSpec;
    myRowId = rowId;
    myItemId = itemId;
    myCause = cause;
  }

  /**
   * Returns the attribute spec of the problematic attribute
   *
   * @return attribute spec
   */
  @NotNull
  public AttributeSpec<?> getAttributeSpec() {
    return myAttributeSpec;
  }

  /**
   * If the problem happened when loading a particular row, returns that row ID
   *
   * @return row ID or null
   */
  @Nullable
  public Long getRowId() {
    return myRowId;
  }

  /**
   * If the problem happened when loading a particular item, returns that item ID
   *
   * @return item ID or null
   */
  @Nullable
  public ItemIdentity getItemId() {
    return myItemId;
  }

  /**
   * If the problem caused an exception to be fired at the attributes system, returns that exception
   *
   * @return caught exception or null
   */
  @Nullable
  public Throwable getCause() {
    return myCause;
  }

  @Override
  public String toString() {
    StringBuilder r = new StringBuilder().append("problem:").append(myAttributeSpec);
    if (myRowId != null) {
      r.append(":row=").append(myRowId);
    }
    if (myItemId != null) {
      r.append(":item=").append(myItemId);
    }
    if (myCause != null) {
      r.append(":").append(myCause.getMessage());
    }
    return r.toString();
  }
}
