package com.almworks.jira.structure.api.structure.history;

import com.almworks.jira.structure.api.error.StructureException;

import java.util.List;

public interface HistoryService {
  void queryActivity(HistoryQuery hq, int limit, HistoryConsumer consumer);

  void getHistoryEntries(long structureId, int versionFrom, int versionTo, HistoryConsumer consumer);

  /**
   * Returns history entries for the specified structure that have versions in the specified range.
   * The range is half-opened on the left side.
   * @param versionFrom minimum version, exclusive
   * @param versionTo maximum version, inclusive                    
   * */
  List<HistoryEntry> getHistoryEntries(long structureId, int versionFrom, int versionTo);

  /**
   * Lightweight version of {@link HistoryService#getHistoryEntries(long, int, int)} which does not
   * fetch changes, works much faster.
   */
  List<HistoryEntry> getHistoryEntriesWithoutChanges(long structureId, int versionFrom, int versionTo);

  /**
   * Returns the {@link HistoryEntry#getVersion() version} of the latest history entry for the specified structure.
   * If there are no history entries for the specified structure, returns 0.
   * */
  int getLatestHistoryEntryVersion(long structureId);

  /**
   * Returns the {@link HistoryEntry#getVersion()} of the earliest history entry which timestamp >= {@param timestamp}
   * If there is no such history entry, returns 0.
   * */
  int getEarliestHistoryEntryVersionGtEqDate(long structureId, long timestamp);

  int getEntryCount(long structureId);

  int getEntryCountGtEqDate(long structureId, long timestamp);

  /**
   * @param structureId structure ID
   * @param version version
   * @return a {@code ForestVersion} containing the full forest and the history entry
   * for the given version
   * @throws StructureException if the structure does not exist or is inaccessible;
   * if there is no such version (including intermittent versions for which there is
   * no history entry)
   */
  ForestVersion getForestVersion(long structureId, int version) throws StructureException;
}
