package com.almworks.jira.structure.api.item;

import com.almworks.jira.structure.api.util.JiraFunc;
import com.almworks.jira.structure.api.util.Limits;
import com.atlassian.annotations.PublicApi;
import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.jira.bc.project.component.ProjectComponent;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.attachment.Attachment;
import com.atlassian.jira.issue.changehistory.ChangeHistory;
import com.atlassian.jira.issue.comments.Comment;
import com.atlassian.jira.issue.customfields.option.Option;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.issue.label.Label;
import com.atlassian.jira.issue.link.IssueLink;
import com.atlassian.jira.issue.priority.Priority;
import com.atlassian.jira.issue.resolution.Resolution;
import com.atlassian.jira.issue.status.Status;
import com.atlassian.jira.issue.worklog.Worklog;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectCategory;
import com.atlassian.jira.project.version.Version;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Locale;

/**
 * <p>This class contains utility methods for checking and creating {@link ItemIdentity}.</p>
 *
 * <p>Typical use cases:</p>
 *
 * <pre>
 *   ItemIdentity itemId = CoreIdentities.issue(10000);
 *   ...
 *   if (CoreIdentities.isIssue(itemId)) {
 *     long issueId = itemId.getLongId();
 *   }
 * </pre>
 */
@PublicApi
public class CoreIdentities {
  public static final String ANONYMOUS_USER_ID = "-";

  public static final ItemIdentity NEW_ISSUE = issue(Long.MAX_VALUE);
  public static final ItemIdentity NEW_FOLDER = folder(Long.MAX_VALUE);
  public static final ItemIdentity NEW_GENERATOR = generator(Long.MAX_VALUE);
  public static final ItemIdentity NEW_EFFECTOR = effector(Long.MAX_VALUE);

  public static final ItemIdentity MISSING_ITEM = ItemIdentity.longId(CoreItemTypes.MISSING, 0);

  private CoreIdentities() {
  }

  public static ItemIdentity user(@Nullable ApplicationUser user) {
    return user(user == null ? null : user.getKey());
  }

  public static ItemIdentity user(@Nullable String userKey) {
    return ItemIdentity.stringId(CoreItemTypes.USER, userKey == null ? ANONYMOUS_USER_ID : userKey);
  }

  public static ItemIdentity group(@NotNull String groupId) {
    return ItemIdentity.stringId(CoreItemTypes.GROUP, groupId);
  }

  public static ItemIdentity group(@NotNull Group group) {
    return group(group.getName());
  }

  public static ItemIdentity project(Project value) {
    return project(value.getId());
  }

  public static ItemIdentity project(long projectId) {
    return ItemIdentity.longId(CoreItemTypes.PROJECT, projectId);
  }

  public static ItemIdentity priority(String id) {
    return ItemIdentity.stringId(CoreItemTypes.PRIORITY, id);
  }

  public static ItemIdentity priority(Priority value) {
    return ItemIdentity.stringId(CoreItemTypes.PRIORITY, value.getId());
  }

  public static ItemIdentity status(String id) {
    return ItemIdentity.stringId(CoreItemTypes.STATUS, id);
  }

  public static ItemIdentity status(Status value) {
    return ItemIdentity.stringId(CoreItemTypes.STATUS, value.getId());
  }

  public static ItemIdentity resolution(String id) {
    return ItemIdentity.stringId(CoreItemTypes.RESOLUTION, id);
  }

  public static ItemIdentity resolution(Resolution value) {
    return ItemIdentity.stringId(CoreItemTypes.RESOLUTION, value.getId());
  }

  public static ItemIdentity label(String label) {
    return ItemIdentity.stringId(CoreItemTypes.LABEL, label);
  }

  public static ItemIdentity label(Label value) {
    return ItemIdentity.stringId(CoreItemTypes.LABEL, value.getLabel());
  }

  public static ItemIdentity issueType(String id) {
    return ItemIdentity.stringId(CoreItemTypes.ISSUETYPE, id);
  }

  public static ItemIdentity issueType(IssueType value) {
    return ItemIdentity.stringId(CoreItemTypes.ISSUETYPE, value.getId());
  }

  public static ItemIdentity issue(long issueId) {
    return ItemIdentity.longId(CoreItemTypes.ISSUE, issueId);
  }

  public static ItemIdentity issue(Issue issue) {
    return issue(issue.getId());
  }

  public static ItemIdentity version(@NotNull Version version) {
    return ItemIdentity.longId(CoreItemTypes.VERSION, version.getId());
  }

  public static ItemIdentity version(long id) {
    return ItemIdentity.longId(CoreItemTypes.VERSION, id);
  }

  public static ItemIdentity versionName(String name) {
    return ItemIdentity.stringId(CoreItemTypes.VERSION_NAME, canonicalVersionName(name));
  }

  public static ItemIdentity versionName(@NotNull Version version) {
    return ItemIdentity.stringId(CoreItemTypes.VERSION_NAME, JiraFunc.CANONICAL_VERSION_NAME.la(version));
  }

  public static ItemIdentity generator(long id) {
    return ItemIdentity.longId(CoreItemTypes.GENERATOR, id);
  }

  public static ItemIdentity effector(long id) {
    return ItemIdentity.longId(CoreItemTypes.EFFECTOR, id);
  }

  public static ItemIdentity loopMarker(long rowId) {
    return ItemIdentity.longId(CoreItemTypes.LOOP_MARKER, rowId);
  }

  public static ItemIdentity component(ProjectComponent component) {
    return ItemIdentity.longId(CoreItemTypes.COMPONENT, component.getId());
  }

  public static ItemIdentity component(long id) {
    return ItemIdentity.longId(CoreItemTypes.COMPONENT, id);
  }

  public static ItemIdentity option(Option option) {
    Long optionId = option.getOptionId();
    return optionId == null ? null : ItemIdentity.longId(CoreItemTypes.CF_OPTION, optionId);
  }

  public static ItemIdentity option(long optionId) {
    return ItemIdentity.longId(CoreItemTypes.CF_OPTION, optionId);
  }

  public static ItemIdentity structure(long structureId) {
    return ItemIdentity.longId(CoreItemTypes.STRUCTURE, structureId);
  }

  public static ItemIdentity sprint(long sprintId) {
    return ItemIdentity.longId(CoreItemTypes.SPRINT, sprintId);
  }

  public static ItemIdentity tempoAccount(int accountId) {
    return ItemIdentity.longId(CoreItemTypes.TEMPO_ACCOUNT, accountId);
  }

  public static ItemIdentity i18nFolder(@NotNull String i18nKey) {
    return ItemIdentity.stringId(CoreItemTypes.FOLDER, i18nKey);
  }

  public static ItemIdentity textFolder(@NotNull String text) {
    return ItemIdentity.stringId(CoreItemTypes.FOLDER, "/" + text);
  }

  public static ItemIdentity folder(long folderId) {
    return ItemIdentity.longId(CoreItemTypes.FOLDER, folderId);
  }

  public static ItemIdentity memo(long memoId) {
    return ItemIdentity.longId(CoreItemTypes.MEMO, memoId);
  }

  public static ItemIdentity sdRequestType(int requestTypeId) {
    return ItemIdentity.longId(CoreItemTypes.REQUEST_TYPE, requestTypeId);
  }

  public static ItemIdentity projectCategory(long projectCategoryId) {
    return ItemIdentity.longId(CoreItemTypes.PROJECT_CATEGORY, projectCategoryId);
  }

  public static ItemIdentity projectCategory(@NotNull ProjectCategory projectCategory) {
    return projectCategory(projectCategory.getId());
  }

  public static ItemIdentity attachment(long id) {
    return ItemIdentity.longId(CoreItemTypes.ATTACHMENT, id);
  }

  public static ItemIdentity attachment(@NotNull Attachment attachment) {
    return ItemIdentity.longId(CoreItemTypes.ATTACHMENT, attachment.getId());
  }

  public static ItemIdentity comment(long id) {
    return ItemIdentity.longId(CoreItemTypes.COMMENT, id);
  }

  public static ItemIdentity comment(@NotNull Comment comment) {
    return ItemIdentity.longId(CoreItemTypes.COMMENT, comment.getId());
  }

  public static ItemIdentity worklog(long id) {
    return ItemIdentity.longId(CoreItemTypes.WORKLOG, id);
  }

  public static ItemIdentity worklog(@NotNull Worklog worklog) {
    return ItemIdentity.longId(CoreItemTypes.WORKLOG, worklog.getId());
  }

  public static ItemIdentity changeHistoryGroup(long id) {
    return ItemIdentity.longId(CoreItemTypes.CHANGE_HISTORY_GROUP, id);
  }

  public static ItemIdentity changeHistoryGroup(@NotNull ChangeHistory changeHistory) {
    return ItemIdentity.longId(CoreItemTypes.CHANGE_HISTORY_GROUP, changeHistory.getId());
  }

  public static ItemIdentity changeHistoryItem(long id) {
    return ItemIdentity.longId(CoreItemTypes.CHANGE_HISTORY_ITEM, id);
  }

  public static ItemIdentity issueLink(long id) {
    return ItemIdentity.longId(CoreItemTypes.ISSUE_LINK, id);
  }

  public static ItemIdentity issueLink(@NotNull IssueLink issueLink) {
    return ItemIdentity.longId(CoreItemTypes.ISSUE_LINK, issueLink.getId());
  }

  public static ItemIdentity issueLinkType(long id) {
    return ItemIdentity.longId(CoreItemTypes.ISSUE_LINK_TYPE, id);
  }

  public static ItemIdentity issueLinkType(@NotNull IssueLink issueLink) {
    return ItemIdentity.longId(CoreItemTypes.ISSUE_LINK_TYPE, issueLink.getId());
  }

  public static boolean isIssue(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.ISSUE);
  }

  public static boolean isProject(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.PROJECT);
  }

  public static boolean isComponent(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.COMPONENT);
  }

  public static boolean isVersion(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.VERSION);
  }

  public static boolean isGenerator(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.GENERATOR);
  }

  public static boolean isEffector(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.EFFECTOR);
  }

  public static boolean isAutomation(ItemIdentity itemId) {
    return isGenerator(itemId) || isEffector(itemId);
  }

  public static boolean isStructure(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.STRUCTURE);
  }

  public static boolean isLoopMarker(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.LOOP_MARKER);
  }

  public static boolean isTempoAccount(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.TEMPO_ACCOUNT);
  }

  public static boolean isVersionName(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.VERSION_NAME);
  }

  public static boolean isAttachment(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.ATTACHMENT);
  }

  public static boolean isComment(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.COMMENT);
  }

  public static boolean isWorklog(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.WORKLOG);
  }

  public static boolean isChangeHistoryGroup(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.CHANGE_HISTORY_GROUP);
  }

  public static boolean isChangeHistoryItem(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.CHANGE_HISTORY_ITEM);
  }

  public static boolean isIssueLink(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.ISSUE_LINK);
  }

  public static boolean isIssueLinkType(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.ISSUE_LINK_TYPE);
  }

  public static boolean isFolder(ItemIdentity itemId) {
    return isItemType(itemId, CoreItemTypes.FOLDER);
  }

  public static boolean isMemo(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.MEMO);
  }

  public static boolean isStatus(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.STATUS);
  }

  public static boolean isUser(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.USER);
  }

  public static boolean isPriority(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.PRIORITY);
  }

  public static boolean isResolution(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.RESOLUTION);
  }

  public static boolean isOption(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.CF_OPTION);
  }

  public static boolean isGroup(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.GROUP);
  }

  public static boolean isSprint(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.SPRINT);
  }

  public static boolean isLabel(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.LABEL);
  }

  public static boolean isIssueType(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.ISSUETYPE);
  }

  public static String canonicalVersionName(String versionName) {
    if (versionName == null) {
      return null;
    } else {
      String version = versionName.trim().toLowerCase(Locale.ROOT);
      return version.length() > Limits.MAX_ITEM_STRING_ID_LENGTH ? version.substring(0, Limits.MAX_ITEM_STRING_ID_LENGTH) : version;
    }
  }

  private static boolean isStringType(@Nullable ItemIdentity itemId, @NotNull String itemType) {
    return isItemType(itemId, itemType) && itemId.isStringId();
  }

  private static boolean isLongType(@Nullable ItemIdentity itemId, @NotNull String itemType) {
    return isItemType(itemId, itemType) && itemId.isLongId();
  }

  private static boolean isItemType(@Nullable ItemIdentity itemId, @NotNull String itemType) {
    return itemId != null && itemType.equals(itemId.getItemType());
  }
}
