package com.almworks.jira.structure.api.effector;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.RowValues;
import com.almworks.jira.structure.api.forest.ForestSpec;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.almworks.jira.structure.api.process.ProgressSink;
import com.almworks.jira.structure.api.util.I18nText;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;

/**
 * Produces {@link com.almworks.jira.structure.api.effect.StoredEffect
 * effect descriptions}, given an item forest and attribute values.
 */
@PublicSpi
public interface EffectorFunction {
  /**
   * <p>The main method in the Effector SPI, which processes an item forest
   * and produces effect descriptions.
   *
   * <p>Before calling this method, Structure takes a snapshot of the forest
   * processed by the current effector invocation (or a sub-forest thereof,
   * depending on the position of the effector row within the forest) and
   * tries to calculate {@link com.almworks.jira.structure.api.attribute.StructureAttributeService#getConsistentAttributeValues(ForestSpec, Function, Collection)
   * consistent} values for all attributes {@link EffectorContext#requireAttribute(AttributeSpec)
   * required} by the effector, for all rows in that forest.
   *
   * <p>Structure then calls this method, which uses the forest and attribute
   * values to generate effect descriptions, returning them via the provided
   * {@link EffectCollector}. Please note that it is the responsibility of the
   * {@link com.almworks.jira.structure.api.effect.EffectProvider effect
   * provider} to check item existence and all necessary permissions, as well as
   * de-duplicate the output and resolve any potential conflicts. Therefore, the
   * effector function itself can often "fire and forget" effects into the
   * collector, with only minimal sanity checks, or none at all. However, if
   * necessary, the function can still report warnings to the user with special
   * {@link com.almworks.jira.structure.api.effect.CoreEffects#emitWarning(I18nText, List)
   * "warning"} effect descriptions.
   *
   * <p>A {@link ProgressSink} is also provided for reporting progress and
   * checking cancellation. The function is supposed to {@link ProgressSink#initialize(int)
   * initialize} the progress sink in the beginning (e.g. with the size of
   * {@code itemForest}, if it processes all rows one by one), {@link ProgressSink#increment()
   * increment} the progress and {@link ProgressSink#isCancelled() check} for
   * cancellation while it works (typically in the body of its main loop), and
   * {@link ProgressSink#complete() report} completion in the end.
   *
   * @param itemForest processed item forest (input)
   * @param rowValues required attribute values (input)
   * @param effectCollector collector for the generated effect descriptions (output)
   * @param progressSink used for reporting progress and checking cancellation (input/output)
   */
  void produceEffects(
    @NotNull ItemForest itemForest, @NotNull RowValues rowValues,
    @NotNull EffectCollector effectCollector, @NotNull ProgressSink progressSink);
}
