package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.row.StructureRow;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.function.BiFunction;

/**
 * <p>Propagate attributes have values calculated down through the hierarchy. The child row depends on the value of its parent.
 * For example, an issue field value may be inherited from the parent unless the issue has its own value in that field.</p>
 *
 * <p>Propagate loader represents a propagate attribute. Propagate loading function for a row {@code R}
 * can calculate values based on:</p>
 * <ul>
 * <li>Already calculated value of the same attribute for the parent row of R;</li>
 * <li>Values of attribute dependencies for the row R (but not its parent or siblings);</li>
 * <li>{@link StructureRow} data for R, its parent and siblings;</li>
 * <li>Context values, if declared with {@link #getContextDependencies}.</li>
 * </ul>
 *
 * <p><strong>The loader must be a stable function.</strong> It must not depend on any other inputs, and should
 * provide the same result for the same inputs every time: {@code V(row) = PROPAGATE(V(parent(row)), dependencies(row))}.</p>
 *
 * <p>Loading one sibling should not depend, imply or affect loading another sibling. Only some of the siblings may be loaded in one request.</p>
 *
 * <p>The value for the parent rows and the dependencies for the current row are guaranteed to be loaded by the time the loading function
 * is called. Note that you cannot get dependency values for the parent row - only for the child rows that are actually being loaded.</p>
 *
 * <h3>Using loader data</h3>
 *
 * <p>Note that sometimes a propagate loader will need to store more data in {@link AttributeValue} than the resulting Java
 * type allows. In that case, use loader data - see {@link AttributeValue#withData}.</p>
 *
 * <h3>Sensitive data control</h3>
 *
 * <p>A propagate, as well as all multi-row loaders, is potentially capable of accessing data calculated for
 * rows that the requesting user is not allowed to see. For example, an allocated budget may be inherited from a larger-scope budgeting
 * parent issue, and a user may not have access to that issue.</p>
 *
 * <p>If the propagated attribute is
 * considered sensitive (according to {@link  com.almworks.jira.structure.api.settings.StructureConfiguration#getAttributeSensitivitySettings()}), then
 * the calls to {@link AttributeLoaderContext#getDependencyValue} will not disclose the hidden value. Instead,
 * the loader function will receive {@link AttributeValue#inaccessible()}.</p>
 *
 * <p>In other words, you should not worry about implementing security checks in a propagate, the system does
 * it for you.</p>
 *
 * <h3>Super-root support</h3>
 *
 * <p>Propagates may calculate a value for the super-root row. (See {@link SuperRootRow}.) This value will then be used as the
 * parent value when calculating values for forest roots.</p>
 *
 * <p>However, that feature is optional and disabled by default. In order to allow loading a value for the super-root, the propagate
 * loader must override {@link #isLoadingSuperRoot()} method.</p>
 *
 * <p>Super-root value is loaded via the usual call to {@link #loadChildren}, with super-root being the child (!), the parent row
 * being null, and the parent row value being {@link AttributeValue#undefined()}.</p>
 *
 * @param <T> type of the loaded value
 *
 * @see AttributeLoader
 * @see AggregateAttributeLoader
 * @see ScanningAttributeLoader
 * @see com.almworks.jira.structure.api.settings.AttributeSensitivitySetting
 */
@PublicSpi
public interface PropagateAttributeLoader<T> extends MultiRowAttributeLoader<T> {
  /**
   * <p>Provider of a loading function for the children of the given parent row. When there's a need to load a propagate value for row R,
   * then first this method is called for the parent of R. Then the function that is received will be called to calculate the value for R and
   * any other siblings that are also requested in this loading process.</p>
   *
   * <p>The implementation can make preparatory calculations and then return a function that will produce results for specific rows. The value
   * of the dependencies for the children rows are available only when that function is called.</p>
   *
   * <p>You should not include costly calculations for all the children rows in the implementation of this method, because {@code loadChildren()}
   * may called even when only one child row needs to be calculated.</p>
   *
   * <p>The implementation must be a stable function - see {@link PropagateAttributeLoader}. Note that {@code loadChildren()} and the functions it
   * produced may be called intermittently without any particular order. In other words, the function produced by {@code loadChildren()} should
   * be working even after another {@code loadChildren()} is called.</p>
   *
   * <p>If for some reason this loader is not applicable for the given row, it should return {@code null}. Note, however, that if multiple propagate
   * loaders work on the same attribute, it could be tricky and lead to unexpected results.</p>
   *
   * @param parentValue the value of this attribute for the parent row, or {@link AttributeValue#undefined()} if there's no parent
   * @param context loading context
   * @return a function that will produce a value for every child row of this parent, given the child row and the loading context, or {@code null}
   * if the values for the children rows of this parent cannot be loaded with this loader
   */
  @Nullable
  BiFunction<StructureRow, PropagateAttributeContext, AttributeValue<T>> loadChildren(
    @NotNull AttributeValue<T> parentValue, @NotNull PropagateAttributeContext.Parent context);

  /**
   * <p>Determines if this loader can provide a value for the super-root row.</p>
   *
   * <p>When this method returns {@code false}, the super-root value will always be {@link AttributeValue#undefined()}. When it is {@code true},
   * the super-root value will be determined by calling {@link #loadChildren} with {@code null} as the parent row and super-root as the child row.</p>
   *
   *  <p>The returned value must be the same throughout the lifetime of the object.</p>
   *
   * @return true if this loader should be called to load the value for the super-root item
   * @see SuperRootRow
   */
  default boolean isLoadingSuperRoot() {
    return false;
  }
}
