package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.ValueFormat;
import com.almworks.jira.structure.api.error.StructureProviderException;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>Attributes extension point - you can add new attributes and attribute implementations to the system by implementing this interface
 * and declaring the implementation in {@code atlassian-plugin.xml}.</p>
 *
 * <p>The declaration is pretty simple - you only need to refer to the class:
 * {@code <structure-attribute-loader-provider key="my-loader" class="com.acme.myapp.MyAttributeLoaderProvider"/>}</p>
 *
 * <p>The class will be instantiated as a singleton component, so you can inject all the required services through the constructor parameters.</p>
 *
 * <p>An implementation of the provider would typically check the passed attribute spec to understand if it can offer a loader
 * for that ID and parameters.</p>
 *
 * <p>The provider will return {@code null} if it cannot offer a loader for the given spec.</p>
 *
 * <h3>Analyzing value format and conversions</h3>
 *
 * <p>Note that the {@link ValueFormat} of the created loader must not necessarily correspond to the value format provided in the
 * attribute spec. The provider should offer the best matching loader. The system will then convert the loaded values, if such conversion
 * is possible.</p>
 *
 * <p>This means that typically loader providers do not analyze the value format. Instead, they would use {@link AttributeSpec#as(ValueFormat)}
 * method on the provided attribute spec to cast it to the format they support.</p>
 *
 * <p>However, when a provider can offer different loaders for different value formats, they would analyze the requested format and offer the
 * best match.</p>
 *
 * <h3>Loader caching</h3>
 *
 * <p>Most attribute loader providers would emit the same attribute loader for the same attribute spec. To speed up attribute loading,
 * the system will cache the attribute loaders per each used attribute spec.</p>
 *
 * <p>The provider may indicate that the loader should not be cached by calling {@link AttributeProviderContext#mustNotCacheLoader()}.
 * This is needed when the provided loader may be different each time.</p>
 *
 * <h3>Using context values</h3>
 *
 * <p>A provider may use the current user or other context values from {@link AttributeContext}, but that will automatically make the
 * loader non-cacheable, and also will implicitly introduce the corresponding context dependency to that loader.</p>
 *
 * <h3>Order</h3>
 *
 * <p>You can define the relative order of a provider by using {@code "order"} attribute in {@code atlassian-plugin.xml}:
 * {@code <structure-attribute-loader-provider key="my-loader" class="com.acme.myapp.MyAttributeLoaderProvider" order="-10"/>}</p>
 *
 * <p>Doing so will affect the order in which attribute loaders for the same attribute spec are called. The loaders created by a provider
 * with a lower order number will be called first. The default order value for all providers is {@code 0}.</p>
 *
 * <p>This means that you can introduce a loader for a specific attribute with a low order value, and that would let it override existing loaders and
 * return a value first.</p>
 *
 * <h3>Concurrency and state</h3>
 *
 * <p>Attribute loader provider may be called concurrently, for the same or different attribute specs. The provider should be a stateless, thread-safe
 * component and not keep any state between the calls.</p>
 *
 * <p>If the attribute implementation does need to keep state between the loadings, consider introducing a separate component responsible for
 * maintaining that state.</p>
 *
 * @see AttributeLoader
 * @see AttributeContext
 */
@PublicSpi
public interface AttributeLoaderProvider {
  /**
   * Creates an attribute loader for the given attribute spec.
   *
   * @param attributeSpec attribute spec
   * @param context creation context
   * @return a loader that can load values for the given spec, or a spec with a different value format; {@code null} if this provider cannot offer
   * a loader
   * @throws StructureProviderException if there was a problem creating a loader (the provider will continue being called)
   */
  @Nullable
  AttributeLoader<?> createAttributeLoader(@NotNull AttributeSpec<?> attributeSpec, @NotNull AttributeProviderContext context)
    throws StructureProviderException;
}
