package com.almworks.jira.structure.api.forest;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.raw.ArrayForest;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import javax.annotation.concurrent.Immutable;

/**
 * <p>ForestChange represents a single change on a forest. A list of ForestChanges represent a difference between two
 * forests. ForestChanges are used only to represent the difference - to optimize event processing, traffic and to
 * provide data to make visual cues about the change. ForestChanges are <strong>not</strong> used to record history or provide
 * additional information about the forest.</p>
 *
 * <p>There are several types of the operations, all defined as the inner classes of {@link ForestChange}:</p>
 *
 * <ul>
 *   <li>{@link Add} - add operation</li>
 *   <li>{@link Move} - move operation</li>
 *   <li>{@link Remove} - remove operation</li>
 *   <li>{@link Reorder} - reorder operation</li>
 * </ul>
 *
 * <p>Note that {@code Reorder} operation can be expressed as a series of {@code Move} operations. Structure can use
 * either one {@code Reorder} or several {@code Move} changes to express reordering.</p>
 *
 * <p>All classes are immutable.</p>
 */
@PublicApi
@Immutable
public abstract class ForestChange {
  private ForestChange() {}

  /**
   * Creates an "add" forest change.
   *
   * @param under the row under which the addition took place, or 0 if added at the top level
   * @param after the previous sibling to the first added row, or 0 if added as a first row under the parent
   * @param added forest that was added
   * @return change
   */
  public static ForestChange add(long under, long after, Forest added) {
    return new Add(added, under, after);
  }

  /**
   * Creates a "move" forest change.
   *
   * @param under the row under which the moved rows were placed
   * @param after the row after which the moved rows were placed
   * @param rows a list of rows that were moved
   * @return change
   */
  public static ForestChange move(long under, long after, LongList rows) {
    return new Move(rows, under, after);
  }

  /**
   * Creates a "remove" forest change.
   *
   * @param rows a list of rows that were removed from the forest
   * @return change
   */
  public static ForestChange remove(LongList rows) {
    return new Remove(rows);
  }

  /**
   * Creates a "reorder" forest change.
   *
   * @param under the row under which the reordering took place, or 0 if added at the top level
   * @param children row IDs of the children, in the desired new order
   * @return change
   */
  public static ForestChange reorder(long under, LongList children) {
    return new Reorder(under, children);
  }

  /**
   * Given mutable forest, apply the change to it.
   *
   * @param forest forest to change
   */
  public abstract void apply(@NotNull ArrayForest forest);
  // todo report non-applicable changes, see 6e086e2a94d66efd19dce4e322e9b35f10b192f6

  /**
   * Apply the change to the visitor
   *
   * @param visitor receiver of the call based on the type of this change
   */
  public abstract void accept(Visitor visitor);


  /**
   * The interface to use when making different actions depending on the specific type of the change.
   *
   * @see ForestChange#accept
   */
  public interface Visitor {
    void visit(Add add);
    void visit(Move move);
    void visit(Remove remove);
    void visit(Reorder reorder);
  }


  /**
   * Represents addition to the forest. An addition of a whole contiguous sub-forest may be expressed with this class.
   * The added rows must not be already present in the forest. (Unlike Structure 2 API, this is not "merge" operation.)
   */
  public static class Add extends ForestChange {
    // todo expand with variations: merge, establish (replace subforest), addOneRow

    @NotNull
    private final Forest myAddedForest;
    private final long myUnder;
    private final long myAfter;

    /**
     * Creates "add" change with a forest being added.
     */
    public Add(@NotNull Forest addedForest, long under, long after) {
      //noinspection ConstantConditions
      if (addedForest == null) {
        throw new IllegalArgumentException("addedForest must not be null");
      }

      myAddedForest = ArrayForest.ensureImmutability(addedForest);
      myUnder = under;
      myAfter = after;
    }

    /**
     * Creates "add" change with one row being added.
     */
    public Add(long row, long under, long after) {
      this(new ArrayForest(LongArray.create(row), IntArray.create(0), true).makeImmutable(), under, after);
    }

    /**
     * Returns the added forest.
     */
    @NotNull
    public Forest getAddedForest() {
      return myAddedForest;
    }

    /**
     * Returns the parent of the (roots of the) added forest, 0 if top level.
     */
    public long getUnder() {
      return myUnder;
    }

    /**
     * Returns the immediately preceding sibling of the first root of the added forest.
     */
    public long getAfter() {
      return myAfter;
    }

    public void apply(@NotNull ArrayForest forest) {
      forest.addForest(myUnder, myAfter, myAddedForest);
    }

    public void accept(Visitor visitor) {
      visitor.visit(this);
    }

    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;

      Add add = (Add) o;

      if (myAfter != add.myAfter) return false;
      if (myUnder != add.myUnder) return false;
      if (!myAddedForest.equals(add.myAddedForest)) return false;

      return true;
    }

    public int hashCode() {
      int result = myAddedForest.hashCode();
      result = 31 * result + (int) (myUnder ^ (myUnder >>> 32));
      result = 31 * result + (int) (myAfter ^ (myAfter >>> 32));
      return result;
    }

    public String toString() {
      return myUnder + "." + myAfter + "+" + myAddedForest;
    }
  }



  /**
   * Represents removal of one or more rows. If the rows have sub-rows, they are removed too. The list of rows
   * may contain rows from under different parent rows.
   */
  public static class Remove extends ForestChange {
    @NotNull
    private final LongList myRemovedRows;

    /**
     * Constructs "remove" change of the given rows.
     */
    public Remove(@NotNull LongList removedRows) {
      //noinspection ConstantConditions
      if (removedRows == null) {
        throw new IllegalArgumentException("removedRows must not be null");
      }
      assert !removedRows.isEmpty();
      // no way to ensure immutability at runtime except by copying
      myRemovedRows = LongArray.copy(removedRows);
    }

    /**
     * Returns the removed rows. All sub-rows of the listed rows will also be removed.
     */
    @NotNull
    public LongList getRemovedRows() {
      return myRemovedRows;
    }

    public void apply(@NotNull ArrayForest forest) {
      for (LongIterator ii : myRemovedRows) {
        forest.removeSubtree(ii.value());
      }
    }

    public void accept(Visitor visitor) {
      visitor.visit(this);
    }

    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;

      Remove remove = (Remove) o;

      if (!myRemovedRows.equals(remove.myRemovedRows))
        return false;

      return true;
    }

    public int hashCode() {
      return myRemovedRows.hashCode();
    }

    public String toString() {
      return "-" + myRemovedRows;
    }
  }


  /**
   * Represents moving one or more rows from their current places in forest under the specified position.
   * The rows may come from under different parents in the forest. They are placed in sequence, according to
   * the list, at (under, after) position.
   */
  public static class Move extends ForestChange {
    @NotNull
    private final LongList myMovedRows;
    private final long myUnder;
    private final long myAfter;

    public Move(@NotNull LongList movedRows, long under, long after) {
      //noinspection ConstantConditions
      if (movedRows == null) {
        throw new IllegalArgumentException("movedRows must not be null");
      }
      // no way to ensure immutability at runtime except by copying
      myMovedRows = LongArray.copy(movedRows);
      myUnder = under;
      myAfter = after;
    }

    /**
     * Returns the moved rows. They are placed in the target position in the order they appear in this list.
     * All sub-rows of these rows are moved along with their parent.
     */
    @NotNull
    public LongList getMovedRows() {
      return myMovedRows;
    }

    /**
     * Returns the new parent of the moved row, 0 if top level.
     */
    public long getUnder() {
      return myUnder;
    }

    /**
     * Returns the new immediately preceding sibling of the first moved row, 0 if none.
     */
    public long getAfter() {
      return myAfter;
    }

    public void apply(@NotNull ArrayForest forest) {
      long after = myAfter;
      for (LongIterator ii : myMovedRows) {
        try {
          forest.moveSubtree(ii.value(), myUnder, after);
        } catch (StructureException e) {
          // todo get rid of exception
        }
        after = ii.value();
      }
    }

    public void accept(Visitor visitor) {
      visitor.visit(this);
    }

    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;

      Move move = (Move) o;

      if (myAfter != move.myAfter) return false;
      if (myUnder != move.myUnder) return false;
      if (!myMovedRows.equals(move.myMovedRows)) return false;

      return true;
    }

    public int hashCode() {
      int result = myMovedRows.hashCode();
      result = 31 * result + (int) (myUnder ^ (myUnder >>> 32));
      result = 31 * result + (int) (myAfter ^ (myAfter >>> 32));
      return result;
    }

    public String toString() {
      return myUnder + "." + myAfter + "<-" + myMovedRows;
    }
  }


  /**
   * Represents a complete reorder of the direct children of a given row. The list of children must contain
   * exactly the same rows, but in a different order.
   */
  public static class Reorder extends ForestChange {
    private final long myUnder;

    @NotNull
    private final LongList myChildren;

    /**
     * Creates a "reorder" change.
     */
    public Reorder(long under, @NotNull LongList children) {
      //noinspection ConstantConditions
      if (children == null) {
        throw new IllegalArgumentException("children must not be null");
      }
      myUnder = under;
      // no way to ensure immutability at runtime except by copying
      myChildren = LongArray.copy(children);
    }

    /**
     * Returns the parent of the reordered rows, 0 if top-level rows are reordered.
     */
    public long getUnder() {
      return myUnder;
    }

    /**
     * Returns the new order of the child elements.
     */
    @NotNull
    public LongList getChildren() {
      return myChildren;
    }

    public void apply(@NotNull ArrayForest forest) {
      forest.reorder(myUnder, myChildren);
    }

    public void accept(Visitor visitor) {
      visitor.visit(this);
    }

    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;

      Reorder reorder = (Reorder) o;

      if (myUnder != reorder.myUnder) return false;
      if (!myChildren.equals(reorder.myChildren)) return false;

      return true;
    }

    public int hashCode() {
      int result = (int) (myUnder ^ (myUnder >>> 32));
      result = 31 * result + myChildren.hashCode();
      return result;
    }

    public String toString() {
      return myUnder + "#" + myChildren;
    }
  }
}
