package com.almworks.jira.structure.api.error;

import com.almworks.jira.structure.api.darkfeature.DarkFeatures;
import com.almworks.jira.structure.api.item.CoreIdentities;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.row.RowManager;
import com.almworks.jira.structure.api.util.*;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Locale;

/**
 * <p>{@code StructureException} can be thrown for different causes that involve Structure plugin. It usually means that the
 * operation that is traced back to the user's action cannot be performed and should be reported as an error.</p>
 *
 * <p>Each exception is associated with a specific value from enumeration {@link StructureError}. Exception may also
 * carry additional message and information about affected structure, view, item or row. The general method
 * {@link #getMessage()}, returns all information included with the exception, except for the stack trace and cause.
 * </p>
 *
 * <h4>Displaying User-Friendly Errors</h4>
 *
 * <p>
 *   To display an error on the user interface: use {@link #getLocalizedMessage()} or {@link #getLocalizedMessage(ApplicationUser)}
 *   to get a human-friendly error in the user's locale. However, in many cases the error message will not be localized
 *   and the result of {@code getLocalizedMessage()} will contain the problem description in English.
 * </p>
 * <p>
 *   So for the best result, you should check {@link #isLocalized()}
 *   method and if exception is not localized, use some wrapper text to display the error to the user in a good way:
 * </p>
 *
 * <pre>
 * Java code:
 *   try {
 *     ...
 *   } catch (StructureException e) {
 *     if (e.isLocalized()) {
 *       setDisplayedError(e.getLocalizedMessage());
 *     } else {
 *       setDisplayedError(getText("my.errors.structure-error", e.getLocalizedMessage()));
 *     }
 *   }
 *</pre>
 *
 * <h4>Throwing StructureException</h4>
 *
 * <p>An instance of {@code StructureException} may be created by using one of the available constructors, but
 * it might be more convenient to start from {@link StructureError} and use chained builder commands, ending with
 * message specification:</p>
 *
 * <pre>
 *   throw StructureErrors.GENERIC_ERROR.withMessage("cannot foo bar");
 *   ...
 *   throw StructureErrors.INVALID_JQL.causedBy(caughtException).forStructure(id).withoutMessage();
 *   ...
 *   throw StructureErrors.VIEW_EDIT_DENIED.forView(id).withLocalizedMessage("error.view.edit.denied", id, name);
 * </pre>
 *
 * @see StructureError
 */
@PublicApi
public class StructureException extends Exception {
  private static final Logger logger = LoggerFactory.getLogger(StructureException.class);

  private static final boolean ITEM_LOOKUP_ENABLED = isItemLookupEnabled();
  private static final ThreadLocal<Boolean> ITEM_LOOKUP_IN_PROGRESS = new ThreadLocal<>();

  @NotNull
  private final StructureError myError;

  @NotNull
  private final String myProblemDetails;

  private final long myStructure;
  private final long myView;
  private final long myRow;
  private final ItemIdentity myItem;

  private final String myMessageKey;
  private final Object[] myMessageParameters;


  /**
   * Constructs an instance of exception.
   *
   * @param error structure error code
   */
  public StructureException(@Nullable StructureError error) {
    this(error, null, null, null, null, null, null, null);
  }

  /**
   * Constructs an instance of exception.
   *
   * @param error structure error code
   * @param message additional message text
   */
  public StructureException(@Nullable StructureError error, @Nullable String message) {
    this(error, null, null, null, null, null, message, null);
  }

  /**
   * Constructs an instance of exception.
   *
   * @param error structure error code
   * @param structure structure in question
   */
  public StructureException(@Nullable StructureError error, @Nullable Long structure) {
    this(error, null, structure, null, null, null, null, null);
  }

  /**
   * Constructs an instance of exception.
   *
   * @param error structure error code
   * @param structure structure in question
   * @param row related row ID
   */
  public StructureException(@Nullable StructureError error, @Nullable Long structure, @Nullable Long row) {
    this(error, null, structure, null, row, null, null, null);
  }

  /**
   * Constructs an instance of exception.
   *
   * @param error structure error code
   * @param structure structure in question
   * @param row related row ID
   * @param message additional message text
   */
  public StructureException(@Nullable StructureError error, @Nullable Long structure, @Nullable Long row, @Nullable String message) {
    this(error, null, structure, null, row, null, message, null);
  }

  /**
   * Constructs an instance of exception. For convenience, use one of the overloaded constructors.
   *
   * @param error structure error code
   * @param structure structure in question
   * @param row related row ID
   * @param message additional message text
   * @param cause throwable cause
   */
  public StructureException(@Nullable StructureError error, @Nullable Long structure, @Nullable Long row, @Nullable String message, @Nullable Throwable cause) {
    this(error, cause, structure, null, row, null, message, null);
  }

  /**
   * Constructs an instance of exception. For convenience, use one of the overloaded constructors.
   *
   * @param error structure error code
   * @param structure structure in question
   * @param row related row ID
   * @param view related view ID
   */
  public StructureException(@Nullable StructureError error, @Nullable Long structure, @Nullable Long row, @Nullable Long view) {
    this(error, null, structure, view, row, null, null, null);
  }

  /**
   * Constructs an instance of exception. For convenience, use one of the overloaded constructors.
   *
   * @param error structure error code
   * @param structure structure in question
   * @param row related row ID
   * @param view related view ID
   * @param message additional message text
   */
  public StructureException(@Nullable StructureError error, @Nullable Long structure, @Nullable Long row, @Nullable Long view, @Nullable String message) {
    this(error, null, structure, view, row, null, message, null);
  }

  /**
   * Constructs an instance of this exception. Not intended to be called directly.
   *
   * @param error structure error code
   * @param cause throwable cause
   * @param structure related structure ID
   * @param view related view ID
   * @param row related row ID
   * @param item related item ID
   * @param message additional message text
   * @param messageKey i18n message text
   * @param messageParameters i18n message parameters
   */
  protected StructureException(StructureError error, @Nullable Throwable cause, @Nullable Long structure,
    @Nullable Long view, @Nullable Long row, @Nullable ItemIdentity item, @Nullable String message, @Nullable String messageKey,
    @Nullable Object... messageParameters)
  {
    super(createMessage(error, structure, view, row, item, message, messageKey, messageParameters), cause);
    myError = error == null ? StructureErrors.GENERIC_ERROR : error;
    // avoid double calculation of root-locale message by providing message that already contains it
    String details = createDetails(message, messageKey, messageParameters);
    myProblemDetails = details.isEmpty() ? "error " + myError : details;
    myStructure = structure == null ? 0L : structure;
    myView = view == null ? 0L : view;
    myRow = row == null ? 0L : row;
    myItem = item;
    myMessageKey = messageKey;
    myMessageParameters = messageParameters;
  }

  private static String createMessage(StructureError error, Long structure, Long view, Long row, ItemIdentity itemId,
    @Nullable String message, @Nullable String messageKey, @Nullable Object... messageParameters)
  {
    StringBuilder b = new StringBuilder();
    if (error == null) error = StructureErrors.GENERIC_ERROR;
    String details = createDetails(message, messageKey, messageParameters);
    if (!details.isEmpty()) b.append(details).append(" - ");
    b.append("structure ").append(error.name()).append(" (code:").append(error.getCode());
    if (structure != null && structure > 0) b.append(" structure:").append(structure);
    if (view != null && view > 0) b.append(" view:").append(view);
    if (row != null && row > 0) {
      b.append(" row:").append(row);
      if (ITEM_LOOKUP_ENABLED) {
        protectFromReentrancy(() -> appendRowDescription(b, row));
      }
    }
    if (itemId != null) {
      b.append(" item:").append(itemId);
      if (ITEM_LOOKUP_ENABLED) {
        protectFromReentrancy(() -> appendItemDescription(b, itemId));
      }
    }
    b.append(')');
    return b.toString();
  }

  @NotNull
  private static String createDetails(String message, String messageKey, Object... messageParameters) {
    if (message == null) {
      message = getRootLocaleMessage(messageKey, messageParameters);
      if (message == null) message = "";
    }
    return message;
  }

  private static void appendRowDescription(StringBuilder b, long rowId) {
    try {
      RowManager rowManager = JiraComponents.getOSGiComponentInstanceOfType(RowManager.class);
      if (rowManager != null) {
        appendItemDescription(b, rowManager.getRow(rowId).getItemId());
      }
    } catch (Exception | LinkageError e) {
      logger.warn("Error on building structure exception", e);
    }
  }

  private static void appendItemDescription(StringBuilder b, ItemIdentity itemId) {
    try {
      String description = StructureUtil.getItemDescription(itemId);
      if (description != null) b.append(" (").append(description).append(')');
    } catch (Exception | LinkageError e) {
      logger.warn("Error on building structure exception", e);
    }
  }

  private static String getRootLocaleMessage(String messageKey, Object... messageParameters) {
    // we could use Locale.ROOT here, but it won't work in development environment because there's no StructureMessages.properties
    try {
      return StructureUtil.getText(Locale.ENGLISH, null, messageKey, messageParameters);
    } catch (Exception | LinkageError e) {
      logger.warn("Error on building structure exception", e);
      // protect from errors and exceptions in the exception initializer
      return "";
    }
  }

  /**
   * Provides string representation of the exception. Overrides default toString(), which uses getLocalizedMessage().
   */
  public String toString() {
    String s = getClass().getName();
    String message = getMessage();
    return message != null ? s + ": " + message : s;
  }

  /**
   * Returns the error value associated with the exception.
   *
   * @return {@link StructureError} enum value
   */
  @NotNull
  public StructureError getError() {
    return myError;
  }

  /**
   * <p>Returns the part of {@link #getMessage()} that corresponds to the original description of the problem.
   * Does not contain structure ID, error code, and related item information, so is more user-friendly than {@link #getMessage()}
   * in case when that information can be more suitably described by the caller.
   * </p>
   *
   * <p>
   *   Normally you shouldn't use this method, use {@link #getLocalizedMessage()} instead. You may want to use this method
   *   if you don't need a potentially localized message, but still need a more user-friendly English message about the problem.
   * </p>
   *
   * @return a more user-friendly error description than {@link #getMessage()} would return.
   */
  @NotNull
  public String getProblemDetails() {
    return myProblemDetails;
  }

  /**
   * Returns related structure, or 0 if no structure is related.
   *
   * @return related structure ID
   */
  public long getStructure() {
    return myStructure;
  }

  /**
   * Returns related view, or 0 if no view is related.
   *
   * @return related view ID
   */
  public long getView() {
    return myView;
  }

  /**
   * Returns related row, or 0 if no row is related.
   *
   * @return related row ID
   */
  public long getRow() {
    return myRow;
  }

  /**
   * Returns related item, or null if no item is related.
   *
   * @return related item ID
   */
  @Nullable
  public ItemIdentity getItem() {
    return myItem;
  }

  /**
   * Checks if there's a i18n message.
   *
   * @return true if there's a localized message
   * @see #getLocalizedMessage()
   */
  public boolean isLocalized() {
    return myMessageKey != null;
  }

  /**
   * Gets the localized message about the problem in the current user's locale. If there's no i18n message available,
   * returns some friendly message, provided by {@link #getProblemDetails()}.
   *
   * @return i18n-ized error message in current user's locale
   * @see #getLocalizedMessage(ApplicationUser)
   */
  @Override
  @NotNull
  public String getLocalizedMessage() {
    return isLocalized() ? StructureUtil.getTextInCurrentUserLocale(myMessageKey, myMessageParameters) : getProblemDetails();
  }

  /**
   * Gets the localized message about the problem in the given user's locale. If there's no i18n message available,
   * returns some friendly message, provided by {@link #getProblemDetails()}.
   *
   * @param user target user
   * @return i18n-ized error message in current user's locale
   * @see #getLocalizedMessage()
   */
  @NotNull
  public String getLocalizedMessage(@Nullable ApplicationUser user) {
    return isLocalized() ? StructureUtil.getText(null, user, myMessageKey, myMessageParameters) : getProblemDetails();
  }

  @NotNull
  public I18nText asI18nText() {
    return isLocalized() ? new I18nText(myMessageKey, myMessageParameters) : new I18nText(getProblemDetails());
  }
  /*
   * We disable item lookup by default because of https://dev.almworks.com/browse/STR-3591
   */
  private static boolean isItemLookupEnabled() {
    // Check the new property
    String enable = DarkFeatures.getProperty("structure.exception.enable.lookup");
    if (enable != null) return Boolean.parseBoolean(enable);
    // Still support the old property - used by tests to avoid exceptions
    String disable = DarkFeatures.getProperty("structure.exception.disable.lookup");
    if (disable != null) return !Boolean.parseBoolean(disable);
    // Otherwise, enable only in dev mode
    return StructureUtil.isDevMode();
  }

  /**
   * This protection is needed if item lookup is enabled. Trying to look up details for a row or an item
   * may result in the same StructureException, thus creating an endless recursion.
   *
   * @see #isItemLookupEnabled()
   * @see <a href="https://dev.almworks.com/browse/STR-3591">STR-3591</a>
   */
  private static void protectFromReentrancy(Runnable code) {
    Boolean value = ITEM_LOOKUP_IN_PROGRESS.get();
    if (value != null && value) return;
    ITEM_LOOKUP_IN_PROGRESS.set(Boolean.TRUE);
    try {
      code.run();
    } finally {
      ITEM_LOOKUP_IN_PROGRESS.remove();
    }
  }


  /**
   * <p>A builder for {@link StructureException}.</p>
   *
   * <p>Usage example:</p>
   *
   * <pre>
   *   throw StructureErrors.GENERIC_ERROR.forStructure(structureId).causedBy(e).withoutMessage();
   * </pre>
   */
  @PublicApi
  public static class Builder {
    private StructureError myError;
    private String myMessage;
    private Long myStructure;
    private Long myView;
    private Long myRow;
    private ItemIdentity myItem;
    private String myMessageKey;
    private Object[] myMessageParameters;
    private Throwable myCause;

    public Builder(StructureError error) {
      myError = error;
    }

    @NotNull
    public StructureException withLocalizedMessage(@Nullable String messageKey, Object... messageParameters) {
      myMessage = messageKey == null ? "" : getRootLocaleMessage(messageKey, messageParameters);
      myMessageKey = messageKey;
      myMessageParameters = messageParameters;
      return build();
    }

    @NotNull
    public StructureException withMessage(@Nullable String message) {
      myMessage = message;
      myMessageKey = null;
      myMessageParameters = null;
      return build();
    }

    @NotNull
    public StructureException withoutMessage() {
      myMessage = null;
      myMessageKey = null;
      myMessageParameters = null;
      return build();
    }

    @NotNull
    public Builder forStructure(@Nullable Long structure) {
      myStructure = structure;
      return this;
    }

    @NotNull
    public Builder forView(@Nullable Long view) {
      myView = view;
      return this;
    }

    @NotNull
    public Builder forRow(@Nullable Long row) {
      myRow = row;
      return this;
    }

    @NotNull
    public Builder forItem(@Nullable ItemIdentity item) {
      myItem = item;
      return this;
    }

    @NotNull
    public Builder forIssue(long issueId) {
      return forItem(CoreIdentities.issue(issueId));
    }

    @NotNull
    public Builder causedBy(Throwable cause) {
      myCause = cause;
      return this;
    }

    @NotNull
    private StructureException build() {
      return new StructureException(myError, myCause, myStructure, myView, myRow, myItem, myMessage, myMessageKey, myMessageParameters);
    }
  }
}
