package com.almworks.jira.structure.api.backup;

import com.almworks.integers.LongSet;
import com.atlassian.annotations.Internal;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

/**
 * <p>This is an extension point for Structure backup/restore mechanism. You need to declare its implementation in
 * {@code structure-backup-provider} module in {@code atlassian-plugin.xml}
 *
 * <p>Example:
 * {@code <structure-backup-provider key="my-backup-provider" class="com.acme.myapp.MyBackupProvider"/>}
 *
 * <p>You should declare only one {@code structure-backup-provider} module in your app. In case of multiple providers
 * only one of them will be used.
 */
@Internal
public interface StructureBackupProvider {
  /**
   * <p>Called during Structure Backup procedure. Use {@code writer} to back up data in XML format.
   * Data is written to {@code extension} section. {@code extension} element has app key and version values as attributes.
   *
   * <p>Data format:
   * <pre>
   * &lt;extensions&gt;
   *   &lt;extension key="com.acme.app" version="1.0"&gt;
   *     writer adds data here...
   *   &lt;/extension&gt;
   *   ...
   * &lt;/extensions&gt;
   * </pre>
   *
   * @param writer backup data writer
   * @param structureIds the collection of structure IDs to back up; {@code null} means all structures
   */
  void backup(@NotNull BackupWriter writer, @Nullable LongSet structureIds);

  /**
   * <p>Called during Structure Restore procedure. {@code reader} points to {@code extension} element related to your app.
   * You can get app version at the time of backup by reading "version" attribute of {@code extension} element.
   *
   * <p>Attempts to read data outside the {@code extension} section will result in an exception.
   *
   * <p>Use {@code errorsReporter} to warn about problems during restore. The errors will be shown on UI after
   * Structure Restore procedure.
   *
   * @param reader backup data reader
   * @param errorsReporter reporter for errors
   */
  void restore(@NotNull BackupReader reader, @NotNull ErrorsReporter errorsReporter);

  /**
   * <p>Called during Migrate Structure procedure.
   *
   * <p>Works the same way as {@link #restore(BackupReader, ErrorsReporter)} but must restore data only for
   * the provided structures.
   *
   * @param reader backup data reader
   * @param backupToStoredStructureIds keys of this map are IDs of structures to migrate as they occur in a backup;
   *                                   values - corresponding structure IDs in the system
   * @param errorsReporter reporter for migration errors
   * @see #restore(BackupReader, ErrorsReporter)
   */
  void migrate(@NotNull BackupReader reader,
    @NotNull Map<Long, Long> backupToStoredStructureIds,
    @NotNull ErrorsReporter errorsReporter);
}
