package com.almworks.jira.structure.api.attribute.loader.composition;

import com.almworks.integers.LongSet;
import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.attribute.loader.*;
import com.almworks.jira.structure.api.attribute.loader.basic.AbstractAttributeLoader;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.row.StructureRow;
import com.almworks.jira.structure.api.util.ConsiderateLogger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;
import java.util.function.Function;

import static com.almworks.jira.structure.api.attribute.loader.AttributeCachingStrategy.*;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableCollection;
import static java.util.stream.Collectors.*;

public abstract class CompositeAttributeLoader<T, L extends AttributeLoader<T>> extends AbstractAttributeLoader<T> {
  private static final Logger logger = LoggerFactory.getLogger(CompositeAttributeLoader.class);
  private static final ConsiderateLogger considerateLogger = new ConsiderateLogger(logger);

  private static final Comparator<AttributeLoader> ASCENDING_BY_TYPES_WEIGHT =
    Comparator.comparingInt(attributeLoader -> LoaderType.getType(attributeLoader).getLoaderWeight());
  private final AttributeCachingStrategy myCachingStrategy;
  private final Set<AttributeSpec<?>> myAttributeDependencies;
  private final Set<AttributeContextDependency> myContextDependencies;
  private final TrailItemSet myGlobalTrail;
  final Collection<L> myLoaders;

  protected CompositeAttributeLoader(AttributeSpec<T> spec, @NotNull Collection<L> loaders) {
    super(spec);
    myLoaders = Collections.unmodifiableCollection(loaders);
    myCachingStrategy = getCompositeCachingStrategy(spec, myLoaders);
    myAttributeDependencies = getCompositeDependencies(myLoaders);
    myContextDependencies = getCompositeContextDependencies(myLoaders);
    myGlobalTrail = getCompositeGlobalTrail(myLoaders);
  }

  @SuppressWarnings("unchecked")//no inference <T> for conversions AttributeLoader<T> -> SpecificAttributeLoader<T> and generic enum support
  public static <T> AttributeLoader<T> create(AttributeSpec<T> spec, List<AttributeLoader<T>> loaders) {
    if (loaders == null || loaders.isEmpty()) {
      throw new IllegalArgumentException("empty or null loaders");
    }
    LoaderType loaderType = LoaderType.getType(Collections.max(loaders, ASCENDING_BY_TYPES_WEIGHT));
    switch (loaderType) {
    case ITEM:
      return new ItemAL(spec, adaptOrSkipNonTargetTypeLoaders(loaderType, loaders));
    case DERIVED:
      return new DerivedAL(spec, adaptOrSkipNonTargetTypeLoaders(loaderType, loaders));
    case SINGLE_ROW:
      return new SingleRowAL(spec, adaptOrSkipNonTargetTypeLoaders(loaderType, loaders));
    case AGGREGATE:
      return new AggregateAL(spec, adaptOrSkipNonTargetTypeLoaders(loaderType, loaders));
    case PROPAGATE:
      return new PropagateAL(spec, adaptOrSkipNonTargetTypeLoaders(loaderType, loaders));
    case SCANNING:
      return new ScanningAL(spec, adaptOrSkipNonTargetTypeLoaders(loaderType, loaders));
    default:
      throw new IllegalArgumentException("unsupported AttributeLoader type");
    }
  }

  private static <T> List<AttributeLoader> adaptOrSkipNonTargetTypeLoaders(LoaderType targetLoaderType, List<AttributeLoader<T>> loaders) {
    Class<? extends AttributeLoader> targetLoaderClass = targetLoaderType.getLoaderClass();
    return loaders.stream()
      .map(AttributeLoaderAdapter.tryAdaptLoader(targetLoaderType))
      .filter(loader -> {
        if (targetLoaderClass.isInstance(loader)) {
          return true;
        }
        logger.warn("Loader {} excluded due to inconsistency with loader type: {}", loader.getClass(), targetLoaderType);
        return false;
      })
      .map(targetLoaderClass::cast)
      .collect(toList());
  }

  private static <T, L extends AttributeLoader<T>> AttributeValue<T> loadValue(@NotNull Collection<L> loaders,
    @NotNull Function<L, AttributeValue<T>> loadFunction)
  {
    return loaders.stream()
      .map(loadFunction)
      .filter(Objects::nonNull)
      .findFirst()
      .orElse(AttributeValue.undefined());
  }

  @NotNull
  public AttributeCachingStrategy getCachingStrategy() {
    return myCachingStrategy;
  }

  @NotNull
  public Set<AttributeSpec<?>> getAttributeDependencies() {
    return myAttributeDependencies;
  }

  @NotNull
  @Override
  public Set<AttributeContextDependency> getContextDependencies() {
    return myContextDependencies;
  }

  @Nullable
  @Override
  public TrailItemSet getGlobalTrail() {
    return myGlobalTrail;
  }

  final AttributeValue<T> loadCompositeValue(Function<L, AttributeValue<T>> loadFunction) {
    return loadValue(myLoaders, loadFunction);
  }

  public static <L extends AttributeLoader<?>> AttributeCachingStrategy getCompositeCachingStrategy(AttributeSpec<?> spec, Collection<L> loaders) {
    Set<AttributeCachingStrategy> strategies = EnumSet.noneOf(AttributeCachingStrategy.class);
    for (L loader : loaders) {
      if (loader.getCachingStrategy() != null) {
        strategies.add(loader.getCachingStrategy());
      }
    }

    if (strategies.contains(MUST_NOT) && strategies.contains(SHOULD)) {
      considerateLogger.warn(spec.toString(), "composite loader has MUST_NOT and SHOULD caching strategies in it simple loaders simultaneously");
    }

    if (strategies.contains(MUST_NOT)) return MUST_NOT;
    if (strategies.contains(SHOULD_NOT)) return SHOULD_NOT;
    if (strategies.contains(SHOULD)) return SHOULD;
    return MAY;
  }

  public static <L extends AttributeLoader<?>> Set<AttributeSpec<?>> getCompositeDependencies(Collection<L> loaders) {
    return loaders.stream()
      .map(AttributeLoader::getAttributeDependencies)
      .filter(Objects::nonNull)
      .flatMap(Set::stream)
      .collect(collectingAndThen(toSet(), Collections::unmodifiableSet));
  }

  public static <L extends AttributeLoader<?>> Set<AttributeContextDependency> getCompositeContextDependencies(Collection<L> loaders) {
    return loaders.stream()
      .map(AttributeLoader::getContextDependencies)
      .filter(Objects::nonNull)
      .flatMap(Set::stream)
      .collect(collectingAndThen(toCollection(() -> EnumSet.noneOf(AttributeContextDependency.class)), Collections::unmodifiableSet));
  }

  public static <L extends AttributeLoader<?>> TrailItemSet getCompositeGlobalTrail(Collection<L> loaders) {
    return loaders.stream()
      .map(AttributeLoader::getGlobalTrail)
      .filter(Objects::nonNull)
      .reduce(TrailItemSet::union)
      .orElse(null);
  }

  public static List<Class<?>> getLoaderClasses(AttributeLoader<?> loader) {
    if (loader == null) return emptyList();
    LinkedHashSet<Class<?>> set = new LinkedHashSet<>();
    traverseLoaderClasses(loader, set);
    return new ArrayList<>(set);
  }

  private static void traverseLoaderClasses(AttributeLoader<?> loader, Set<Class<?>> set) {
    if (!(loader instanceof CompositeAttributeLoader)) {
      set.add(loader.getClass());
    } else {
      Collection<? extends AttributeLoader<?>> loaders = ((CompositeAttributeLoader<?, ?>) loader).myLoaders;
      for (AttributeLoader<?> elementLoader : loaders) {
        if (elementLoader instanceof AttributeLoaderAdapter) {
          AttributeLoader<?> adaptedLoader = ((AttributeLoaderAdapter<?, ?>) elementLoader).getAdaptedLoader();
          traverseLoaderClasses(adaptedLoader, set);
        } else {
          traverseLoaderClasses(elementLoader, set);
        }
      }
    }
  }


  private static abstract class RowAL<T, L extends RowAttributeLoader<T>> extends CompositeAttributeLoader<T, L> implements RowAttributeLoader<T> {
    private final boolean myWholeForestDependent;

    RowAL(AttributeSpec<T> spec, @NotNull Collection<L> loaders) {
      super(spec, loaders);
      myWholeForestDependent = loaders.stream().anyMatch(RowAttributeLoader::isWholeForestDependent);
    }

    @Override
    public void preload(@NotNull LongSet rowIds, @NotNull ItemForest forest, @NotNull AttributeContext context) {
      for (L loader : myLoaders) {
        loader.preload(rowIds, forest, context);
      }
    }

    @Override
    public boolean isWholeForestDependent() {
      return myWholeForestDependent;
    }
  }


  private static class PropagateAL<T> extends RowAL<T, PropagateAttributeLoader<T>> implements PropagateAttributeLoader<T> {
    PropagateAL(AttributeSpec<T> spec, @NotNull Collection<PropagateAttributeLoader<T>> loaders) {
      super(spec, loaders);
    }

    @Nullable
    @Override
    public BiFunction<StructureRow, PropagateAttributeContext, AttributeValue<T>> loadChildren(@NotNull AttributeValue<T> parentValue,
      @NotNull PropagateAttributeContext.Parent context)
    {
      return myLoaders.stream()
        .map(loader -> loader.loadChildren(parentValue, context))
        .filter(Objects::nonNull)
        .findFirst()
        .orElse(null);
    }

    @Override
    public boolean isLoadingSuperRoot() {
      return myLoaders.stream().anyMatch(PropagateAttributeLoader::isLoadingSuperRoot);
    }
  }


  private static class AggregateAL<T> extends RowAL<T, AggregateAttributeLoader<T>> implements AggregateAttributeLoader<T> {
    AggregateAL(AttributeSpec<T> spec, @NotNull Collection<AggregateAttributeLoader<T>> loaders) {
      super(spec, loaders);
    }

    @Override
    public AttributeValue<T> loadValue(List<AttributeValue<T>> childrenValues, AggregateAttributeContext context) {
      return loadCompositeValue(loader -> loader.loadValue(childrenValues, context));
    }
  }


  private static class ScanningAL<T> extends RowAL<T, ScanningAttributeLoader<T>> implements ScanningAttributeLoader<T> {
    ScanningAL(AttributeSpec<T> spec, @NotNull Collection<ScanningAttributeLoader<T>> loaders) {
      super(spec, loaders);
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull AttributeValue<T> precedingValue, @NotNull ScanningAttributeContext context) {
      return loadCompositeValue(loader -> loader.loadValue(precedingValue, context));
    }
  }


  private static class SingleRowAL<T> extends RowAL<T, SingleRowAttributeLoader<T>> implements SingleRowAttributeLoader<T> {
    SingleRowAL(AttributeSpec<T> spec, @NotNull Collection<SingleRowAttributeLoader<T>> loaders) {
      super(spec, loaders);
    }

    @Override
    public AttributeValue<T> loadValue(StructureRow row, SingleRowAttributeContext context) {
      return loadCompositeValue(loader -> loader.loadValue(row, context));
    }
  }


  private static class DerivedAL<T> extends CompositeAttributeLoader<T, DerivedAttributeLoader<T>> implements DerivedAttributeLoader<T> {
    DerivedAL(AttributeSpec<T> spec, @NotNull Collection<DerivedAttributeLoader<T>> loaders) {
      super(spec, loaders);
    }

    @Override
    public AttributeValue<T> loadValue(DerivedAttributeContext context) {
      return loadCompositeValue(loader -> loader.loadValue(context));
    }
  }


  private static class ItemAL<T> extends CompositeAttributeLoader<T, ItemAttributeLoader<T>> implements ItemAttributeLoader<T> {
    private final Map<String, Collection<ItemAttributeLoader<T>>> myLoadersByItemTypes;

    ItemAL(AttributeSpec<T> spec, @NotNull Collection<ItemAttributeLoader<T>> loaders) {
      super(spec, loaders);
      myLoadersByItemTypes = new ConcurrentHashMap<>();
    }

    @Override
    public boolean isItemTypeSupported(String itemType) {
      return getLoadersToWorkWithItemType(itemType).size() > 0;
    }

    @Override
    public void preload(@NotNull Collection<ItemIdentity> itemIds, @NotNull AttributeContext context) {
      for (ItemAttributeLoader<T> loader : myLoaders) {
        Collection<ItemIdentity> supportedItemIds = itemIds.stream().filter(id -> loader.isItemTypeSupported(id.getItemType())).collect(toList());
        loader.preload(supportedItemIds, context);
      }
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(ItemIdentity itemId, ItemAttributeContext context) {
      return CompositeAttributeLoader.loadValue(getLoadersToWorkWithItemType(itemId.getItemType()), loader -> loader.loadValue(itemId, context));
    }

    private Collection<ItemAttributeLoader<T>> getLoadersToWorkWithItemType(String itemType) {
      Collection<ItemAttributeLoader<T>> attributeLoadersForItemType = myLoadersByItemTypes.get(itemType);
      if (attributeLoadersForItemType != null) {
        return attributeLoadersForItemType;
      }
      return myLoadersByItemTypes.computeIfAbsent(itemType, itemTypeKey -> {
        Collection<ItemAttributeLoader<T>> loadersForItemType = new LinkedList<>();
        for (ItemAttributeLoader<T> loader : myLoaders) {
          if (loader.isItemTypeSupported(itemTypeKey)) {
            loadersForItemType.add(loader);
          }
        }
        return loadersForItemType.isEmpty() ? emptyList() : unmodifiableCollection(loadersForItemType);
      });
    }
  }
}
