package com.almworks.jira.structure.api.util;

import com.almworks.integers.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Array;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * <p><code>La</code> is a utility generic-purpose class for functional expressions.</p>
 *
 * @author Igor Sereda
 */
public abstract class La<T, R> implements Function<T, R>, Predicate<T> {
  private static final La SELF = new La() {
    public Object la(Object argument) {
      return argument;
    }
  };
  private static final La TRUE = new La() {
    public Object la(Object argument) {
      return Boolean.TRUE;
    }
  };
  private static final La FALSE = new La() {
    public Object la(Object argument) {
      return Boolean.FALSE;
    }
  };
  private static final La NOT_NULL = new La<Object, Boolean>() {
    public Boolean la(Object argument) {
      return argument != null;
    }
  };

  private static final La TO_STRING = new La<Object, String>(String.class) {
    @Override
    public String la(Object argument) {
      return argument == null ? null : argument.toString();
    }
  };

  private final Class<R> rClass;

  protected La() {
    rClass = null;
  }

  /**
   * Use this constructor to have class-aware instance, which is needed only for
   * few methods like {@link #array}.
   *
   * @param rClass the class of the result
   */
  protected La(Class<R> rClass) {
    this.rClass = rClass;
  }

  public abstract R la(T argument);

  // ============== constructors ================

  @NotNull
  public static <T, R> La<T, R> adapt(final Function<T, R> f) {
    if (f == null) throw new NullPointerException();
    return new La<T, R>() {
      @Override
      public R la(T argument) {
        return f.apply(argument);
      }
    };
  }

  @NotNull
  public static <T, R> La<T, R> constant(final R result) {
    if (result == Boolean.TRUE) return (La<T, R>) TRUE;
    if (result == Boolean.FALSE) return (La<T, R>) FALSE;
    return new La<T, R>() {
      public R la(T argument) {
        return result;
      }
    };
  }
  
  public static <T> La<T, Boolean> constantFalse() {
    return (La<T, Boolean>)FALSE;
  }

  public static <T> La<T, Boolean> constantTrue() {
    return (La<T, Boolean>)TRUE;
  }

  @NotNull
  public static <T> La<T, Boolean> isEqual(@Nullable final T value) {
    final boolean isNull = value == null;
    return new La<T, Boolean>() {
      public Boolean la(T argument) {
        return isNull ? argument == null : value.equals(argument);
      }
    };
  }

  // we could declare the parameter as Collection<? super T>, but then we'd lose type inference for the calling line
  @NotNull
  public static <T> La<T, Boolean> inCollection(@Nullable final Collection<T> collection) {
    if (collection == null) return constant(false);
    return new La<T, Boolean>() {
      public Boolean la(T argument) {
        return collection.contains(argument);
      }
    };
  }
  
  @NotNull
  public static <T> La<T, Boolean> instanceOf(@Nullable final Class<? extends T> instanceClass) {
    if (instanceClass == null) return constant(false);
    return new La<T, Boolean>() {
      public Boolean la(T argument) {
        return instanceClass.isInstance(argument);
      }
    };
  }

  @NotNull
  public static <T> La<Object, T> cast(@Nullable Class<T> clazz) {
    if (clazz == null) return constant(null);
    return new La<Object, T>() {
      @Override
      public T la(Object argument) {
        return clazz.isInstance(argument) ? clazz.cast(argument) : null;
      }
    };
  }

  @NotNull
  public static <A, B, C> La<A, C> compose(final La<B, C> g, final La<A, ? extends B> f) {
    if (f == null) throw new NullPointerException();
    if (g == null) throw new NullPointerException();
    return new La<A, C>(g.rClass) {
      public C la(A argument) {
        return g.la(f.la(argument));
      }
    };
  }

  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> La<T, T> self() {
    return (La<T, T>) SELF;
  }

  @NotNull
  public static <T, R> La<T, R> fromMap(final Map<? super T, ? extends R> map, Class<R> rClass) {
    if (map == null) throw new NullPointerException();
    return new La<T, R>(rClass) {
      @Override
      public R la(T argument) {
        return map.get(argument);
      }
    };
  }


  // ============== utilities ================

  @Override
  public final R apply(T from) {
    return la(from);
  }

  @Override
  public boolean test(T input) {
    return accepts(input);
  }

  @NotNull
  public <A> La<A, R> apply(final La<A, ? extends T> f) {
    return compose(this, f);
  }

  @NotNull
  public <A> La<T, A> supply(final La<? super R, A> g) {
    return compose(g, this);
  }

  @NotNull
  public <D extends T> ArrayList<D> filter(@Nullable Iterable<D> collection) {
    ArrayList<D> r = new ArrayList<>();
    if (collection != null) {
      for (D item : collection) {
        if (accepts(item)) {
          r.add(item);
        }
      }
    }
    return r;
  }

  @NotNull
  public <D extends T> Iterator<D> filter(@Nullable Iterator<D> iterator) {
    return iterator == null ? Collections.emptyIterator() : new FilterIterator<>(iterator);
  }

  @NotNull
  public <D extends T> Iterable<D> filterIterable(@Nullable final Iterable<D> iterable) {
    if (iterable == null) {
      return Collections.emptyList();
    }
    return () -> new FilterIterator<>(iterable.iterator());
  }

  public boolean accepts(T value) {
    return acceptsResult(la(value));
  }

  private boolean acceptsResult(R value) {
    return value != null && !value.equals(false) && !value.equals(0) && !value.equals(0L);
  }

  @NotNull
  public HashSet<R> hashSet(@Nullable Collection<? extends T> from) {
    if (from == null) return new HashSet<>();
    return addTo(from, new HashSet<>(from.size()), true);
  }

  @NotNull
  public LinkedHashSet<R> linkedHashSet(@Nullable Collection<? extends T> from) {
    if (from == null) return new LinkedHashSet<>();
    return addTo(from, new LinkedHashSet<>(from.size()), true);
  }

  @NotNull
  public List<R> arrayList(@Nullable T ... from) {
    return from == null || from.length == 0 ? Collections.emptyList() : arrayList(Arrays.asList(from));
  }

  public List<R> arrayList(@Nullable Collection<? extends T> from) {
    return arrayList(from, true);
  }

  @NotNull
  public Iterable<R> iterable(@Nullable final Iterable<? extends T> from) {
    return () -> La.this.iterator(from);
  }

  @NotNull
  public Iterator<R> iterator(Iterable<? extends T> from) {
    final Iterator<? extends T> source = from == null ? null : from.iterator();
    if (source == null) return Collections.emptyIterator();
    return new Iterator<R>() {
      public boolean hasNext() {
        return source.hasNext();
      }

      public R next() {
        return la(source.next());
      }

      public void remove() {
        source.remove();
      }
    };
  }

  @NotNull
  public List<R> arrayList(@Nullable Collection<? extends T> from, boolean acceptFalsy) {
    if (from == null) return new ArrayList<>();
    return addTo(from, new ArrayList<>(from.size()), acceptFalsy);
  }
  
  public List<R> arrayList(@Nullable Iterator<? extends T> from) {
    return arrayList(from, true);
  }

  public List<R> arrayList(@Nullable Iterator<? extends T> from, boolean acceptFalsy) {
    if (from == null) return new ArrayList<>();
    return addTo(from, new ArrayList<>(), acceptFalsy);
  }

  @NotNull
  public static <T> LongList longList(@Nullable Collection<T> collection, @NotNull La<? super T, Long> la) {
    if (collection == null || collection.isEmpty()) {
      return LongList.EMPTY;
    }
    LongArray result = new LongArray(collection.size());
    for (T element : collection) {
      Long value = la.la(element);
      if (value != null && value > 0) {
        result.add(value);
      }
    }
    return result;
  }

  @NotNull
  public static <T> IntList intList(@Nullable Collection<T> collection, @NotNull La<? super T, Integer> la) {
    if (collection == null || collection.isEmpty()) {
      return IntList.EMPTY;
    }
    IntArray result = new IntArray(collection.size());
    for (T element : collection) {
      Integer value = la.la(element);
      if (value != null && value > 0) {
        result.add(value);
      }
    }
    return result;
  }

  @NotNull
  public <C extends Collection<R>> C addTo(@Nullable Collection<? extends T> from, @NotNull C to, boolean acceptFalsy) {
    if (from != null) {
      addTo(from.iterator(), to, acceptFalsy);
    }
    return to;
  }

  @NotNull
  public <C extends Collection<R>> C addTo(@Nullable Iterator<? extends T> from, @NotNull C to, boolean acceptFalsy) {
    if (from != null) {
      while (from.hasNext()) {
        T t = from.next();
        R r = la(t);
        if (acceptFalsy || acceptsResult(r)) {
          to.add(r);
        }
      }
    }
    return to;
  }
  
  public <D extends T> Iterator<R> transform(Iterator<D> iterator) {
    return new TransformIterator<>(iterator);
  }
  
  public <D extends T> Iterable<R> transformIterable(final Iterable<D> iterable) {
    return () -> new TransformIterator<>(iterable.iterator());
  }

  @NotNull
  public R[] array(@Nullable Collection<? extends T> from) {
    return array(from, true);
  }

  @NotNull
  public R[] array(@Nullable Collection<? extends T> from, boolean acceptFalsy) {
    Class<R> cls = rClass;
    if (cls == null) {
      throw new UnsupportedOperationException("cannot create array without rClass (" + from + ")");
    }
    R[] result = (R[]) Array.newInstance(cls, from == null ? 0 : from.size());
    if (from != null) {
      int i = 0;
      for (T t : from) {
        if (i >= result.length) {
          throw new IllegalStateException("collection changed on the go");
        }
        R r = la(t);
        if (acceptFalsy || acceptsResult(r)) {
          result[i++] = r;
        }
      }
      if (i < result.length) {
        result = Arrays.copyOf(result, i);
      }
    }
    return result;
  }

  @NotNull
  public Map<R, T> mapInto(@Nullable Collection<? extends T> fromCollection, @NotNull Map<R, T> toMap) {
    return mapInto(fromCollection, toMap, La.<T>self());
  }

  @NotNull
  public <V, C extends T> Map<R, V> mapInto(@Nullable Collection<C> fromCollection, @NotNull Map<R, V> toMap, @NotNull La<? super C, ? extends V> valueFunction) {
    if (fromCollection != null) {
      for (C value : fromCollection) {
        R key = la(value);
        if (key != null) {
          toMap.put(key, valueFunction.la(value));
        }
      }
    }
    return toMap;
  }

  @NotNull
  public <C extends T> Map<R, C> hashMap(@Nullable Collection<C> collection) {
    return hashMap(collection, La.self());
  }

  @NotNull
  public <V, C extends T> Map<R, V> hashMap(@Nullable Collection<C> collection, @NotNull La<? super C, V> valueFunction) {
    HashMap<R, V> map = new HashMap<>(collection == null ? 0 : collection.size());
    return mapInto(collection, map, valueFunction);
  }

  @NotNull
  public <V, C extends T> Map<R, V> linkedHashMap(@Nullable Collection<C> collection, @NotNull La<? super C, V> valueFunction) {
    LinkedHashMap<R, V> map = new LinkedHashMap<>(collection == null ? 0 : collection.size());
    return mapInto(collection, map, valueFunction);
  }

  public int indexOf(@Nullable List<? extends T> list, R sample) {
    if (list == null) return -1;
    for (int i = 0; i < list.size(); i++) {
      R v = la(list.get(i));
      if (sample == null && v == null || sample != null && sample.equals(v)) return i;
    }
    return -1;
  }

  public int lastIndexOf(@Nullable List<? extends T> list, R sample) {
    if (list == null) return -1;
    for (int i = list.size() - 1; i >= 0; --i) {
      R v = la(list.get(i));
      if (sample == null && v == null || sample != null && sample.equals(v)) return i;      
    }
    return -1;
  }

  @Nullable
  public <X extends T> X find(@Nullable Collection<X> collection, R sample) {
    if (collection == null || collection.isEmpty()) return null;
    for (X value : collection) {
      R r = la(value);
      if (r == null && sample == null) {
        return value;
      } else if (r != null && r.equals(sample)) {
        return value;
      }
    }
    return null;
  }
  
  public boolean any(@Nullable Iterable<? extends T> sequence) {
    if (sequence == null) return false;
    for (T value : sequence) if (accepts(value)) return true;
    return false;
  }

  @NotNull
  public La<T, Boolean> not() {
    final La<T, R> positive = this;
    return new La<T, Boolean>() {
      public Boolean la(T argument) {
        return !positive.accepts(argument);
      }
    };
  }

  public La<T, Boolean> and(final La<? super T, ?> second) {
    final La<T, ?> first = this;
    return new La<T, Boolean>() {
      public Boolean la(T argument) {
        return first.accepts(argument) && second.accepts(argument);
      }
    };
  }

  public La<T, R> memoize() {
    final La<T, R> delegate = this;
    final Map<T, R> map = new HashMap<>();
    return new La<T, R>() {
      @Override
      public R la(T argument) {
        R result = map.get(argument);
        if (result == null && !map.containsKey(argument)) {
          result = delegate.la(argument);
          map.put(argument, result);
        }
        return result;
      }
    };
  }

  public La<T, R> memoizeConcurrent() {
    final La<T, R> delegate = this;
    final ConcurrentMap<T, R> map = new ConcurrentHashMap<>();
    return new La<T, R>() {
      @Override
      public R la(T argument) {
        assert argument != null;
        R result = map.get(argument);
        if (result == null) {
          result = delegate.la(argument);
          assert result != null;
          map.putIfAbsent(argument, result);
        }
        return result;
      }
    };
  }

  public Comparator<T> comparator(Comparator<? super R> comparator) {
    return new LaComparator<>(this, comparator);
  }

  public static <A, B extends Comparable<B>> Comparator<A> comparator(La<A, B> function) {
    return function.comparator(new ComparableComparator<>());
  }

  public <C extends Iterable<? extends T>> La<C, List<R>> liftToList() {
    final La<T, R> transform = this;
    return new La<C, List<R>>() {
      @Override
      public List<R> la(C argument) {
        if (argument == null) {
          return null;
        }
        Iterator<? extends T> it = argument.iterator();
        if (it.hasNext()) {
          return transform.arrayList(it);
        } else {
          return Collections.emptyList();
        }
      }
    };
  }

  public static <T> La<T, Boolean> notNull() {
    return NOT_NULL;
  }

  public static <T> La<T, String> stringValue() {
    return TO_STRING;
  }

  public static <A, B> La<A, B> notNull(final La<A, B> la, final B onNull) {
    return new La<A, B>(la.rClass) {
      @Override
      public B la(A argument) {
        B result = la.la(argument);
        return result == null ? onNull : result;
      }
    };
  }

  public Collection<R> image(Collection<T> set) {
    return image(set, false);
  }

  private Collection<R> image(final Collection<T> set, final boolean includeFalsy) {
    if (set == null) return Collections.emptyList();
    return new AbstractCollection<R>() {
      @NotNull
      public Iterator<R> iterator() {
        Iterator<R> it = transform(set.iterator());
        if (!includeFalsy) {
          it = La.self().filter(it);
        }
        return it;
      }

      public int size() {
        if (includeFalsy) {
          return set.size();
        }
        int count = 0;
        for (T v : set) {
          if (accepts(v)) {
            count++;
          }
        }
        return count;
      }
    };
  }


  public static class LaComparator<A, B> implements Comparator<A> {
    private final La<A, B> myFunction;
    private final Comparator<? super B> myComparator;

    public LaComparator(La<A, B> function, Comparator<? super B> comparator) {
      if (function == null) {
        throw new IllegalArgumentException("function is null");
      }
      if (comparator == null) {
        throw new IllegalArgumentException("comparator is null");
      }
      myFunction = function;
      myComparator = comparator;
    }

    @Override
    public int compare(A o1, A o2) {
      B b1 = myFunction.apply(o1);
      B b2 = myFunction.apply(o2);
      return myComparator.compare(b1, b2);
    }
  }

  public static class ComparableComparator<B extends Comparable<B>> implements Comparator<B> {
    @Override
    public int compare(B b1, B b2) {
      if (b1 == null) {
        return b2 == null ? 0 : -1;
      }
      if (b2 == null) {
        return 1;
      }
      return b1.compareTo(b2);
    }
  }
  
  
  public class FilterIterator<D extends T> implements Iterator<D> {
    private final Iterator<D> myOuter;
    private Boolean hasNext = null;
    private D next;

    public FilterIterator(Iterator<D> iterator) {
      myOuter = iterator;
    }

    @Override
    public boolean hasNext() {
      if (hasNext == null) {
        hasNext = calcHasNext();
      }
      return hasNext;
    }

    private boolean calcHasNext() {
      while (myOuter.hasNext()) {
        next = myOuter.next();
        if (accepts(next)) return true;
      }
      return false;
    }

    @Override
    public D next() {
      if (!hasNext())
        throw new NoSuchElementException();
      hasNext = null;
      return next;
    }

    @Override
    public void remove() {
      throw new UnsupportedOperationException();
    }
  }
  
  
  public class TransformIterator<D extends T> implements Iterator<R> {
    private final Iterator<D> mySourceIterator;

    public TransformIterator(Iterator<D> iterator) {
      mySourceIterator = iterator;
    }

    @Override
    public boolean hasNext() {
      return mySourceIterator.hasNext();
    }

    @Override
    public R next() {
      return la(mySourceIterator.next());
    }

    @Override
    public void remove() {
      mySourceIterator.remove();
    }
  }
}
