package com.almworks.jira.structure.api.darkfeature;

import com.almworks.jira.structure.api.util.ConsiderateLogger;
import com.almworks.jira.structure.api.util.JiraComponents;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.config.properties.ApplicationProperties;
import org.jetbrains.annotations.*;
import org.slf4j.LoggerFactory;

import static com.almworks.jira.structure.api.util.StructureUtil.*;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

/**
 * <p>Provides access to user-defined "dark" features and fine-tuning parameters.
 *
 * <p>The idea is to simplify property configuration (for example, have the same configuration on all nodes in cluster environment), so
 * {@link DarkFeatures} aims to replace {@link System#getProperty(String)} and similar method calls. To provide backward compatibility,
 * system property values are used if property is not configured.
 *
 * <p>Many components need these values during initialization phase (for example, to tune cache size). We store "dark" features in Jira's
 * {@link ApplicationProperties} to ensure that they can be accessed before ActiveObjects or any other plugin component is initialized.
 *
 * @see System#getProperty(String)
 * @see System#getProperty(String, String)
 * @see Long#getLong(String, long)
 * @see Integer#getInteger(String, int)
 * @see Boolean#getBoolean(String)
 * @see com.almworks.jira.structure.api.util.StructureUtil#getBooleanSystemProperty(String, boolean)
 */
@PublicApi
public class DarkFeatures {
  private static final ConsiderateLogger logger = new ConsiderateLogger(LoggerFactory.getLogger(DarkFeatures.class));

  public static long getLong(@NotNull String propertyName, long defaultValue) {
    String value = getApplicationProperty(propertyName);
    return isNotEmpty(value) ? lv(value, defaultValue) : Long.getLong(propertyName, defaultValue);
  }

  public static int getInteger(@NotNull String propertyName, int defaultValue) {
    String value = getApplicationProperty(propertyName);
    return isNotEmpty(value) ? iv(value, defaultValue) : Integer.getInteger(propertyName, defaultValue);
  }

  public static boolean getBoolean(@NotNull String propertyName) {
    return getBoolean(propertyName, false);
  }

  public static boolean getBoolean(@NotNull String propertyName, boolean defaultValue) {
    String value = getApplicationProperty(propertyName);
    return isNotEmpty(value) ? Boolean.parseBoolean(value) : getBooleanSystemProperty(propertyName, defaultValue);
  }

  @Nullable
  public static String getProperty(@NotNull String propertyName) {
    return getProperty(propertyName, null);
  }

  @Contract("_, !null -> !null")
  public static String getProperty(@NotNull String propertyName, @Nullable String defaultValue) {
    String value = getApplicationProperty(propertyName);
    return isNotEmpty(value) ? value : System.getProperty(propertyName, defaultValue);
  }

  @Nullable
  @Internal
  public static String getApplicationProperty(@NotNull String propertyName) {
    try {
      ApplicationProperties applicationProperties = JiraComponents.getApplicationProperties();
      return  applicationProperties == null ? null : applicationProperties.getString(propertyName);
    } catch (LinkageError | Exception e) {
      logger.warn("DarkFeatures:", "failed to get feature value from application properties", e);
      return null;
    }
  }
}
