package com.almworks.jira.structure.api.cache.access;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.item.ItemIdentitySet;
import com.atlassian.annotations.Internal;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

/**
 * Provides per-row, per-issue caching and functions to verify row/issue visibility to the user
 */
@Internal
public interface ForestAccessCache {
  void clear();

  void clear(ApplicationUser user);

  /**
   * Calculates/retrieves the set of row IDs (among the passed {@code rows}), which contain items not visible to
   * the specified user.
   *
   * @param rows Rows to check -- only inserted rows must be passed here. Temporary rows (from ItemForest) are not allowed,
   * because this method works with RowManager and caches.
   */
  @NotNull
  LongSet getInvisibleRows(LongIterable rows, String userKey) throws UserLookupException;

  /**
   * @see #getInvisibleRows(LongIterable, String)
   */
  @NotNull
  LongSet getInvisibleRows(LongIterable rows, ApplicationUser user);

  /**
   * Collects invisible rows from a temporary forest. Rows do not have to exist, and rows cache is not used by this
   * method. (Issue cache *is* used, after rows get converted to items.)
   *
   * @param forest a forest to check
   * @param userKey the user
   * @param invisibleRowsCollector collector that will receive row IDs for the rows that contain an item not visible to the user
   * @throws UserLookupException
   */
  void collectInvisibleRows(ItemForest forest, String userKey, LongCollector invisibleRowsCollector) throws UserLookupException;

  void collectInvisibleRows(ItemForest forest, ApplicationUser user, LongCollector invisibleRowsCollector);

  /**
   * Collects issues that are not visible to the specified user. Uses per-user visibility cache to speed up processing.
   *
   * @param issues issue IDs (in any order)
   * @param user the user
   * @param overrideSecurity when true, searching with overrideSecurity is used
   * @param invisibleIssuesCollector a collector to receive invisible issue IDs
   */
  void collectInvisibleIssues(LongIterable issues, ApplicationUser user, boolean overrideSecurity,
    LongCollector invisibleIssuesCollector);

  ItemIdentitySet getInvisibleItems(Set<ItemIdentity> items, ApplicationUser user);

  class UserLookupException extends Exception {
  }
}
