package com.almworks.jira.structure.api.attribute.loader.builder;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.attribute.loader.*;
import com.almworks.jira.structure.api.attribute.loader.basic.AbstractDerivedAttributeLoader;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.google.common.collect.ImmutableSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

public class BiDerivedAttributeLoaderBuilder<T, X, Y> extends AttributeLoaderBuilder<T, BiDerivedAttributeLoaderBuilder<T, X, Y>> {
  private AttributeSpec<X> myXDependencySpec;
  private AttributeSpec<Y> myYDependencySpec;
  private BiFunction<X, Y, T> myValueFunction;
  private Function<X, ItemIdentity> myXTrailFunction;
  private Function<Y, ItemIdentity> myYTrailFunction;
  private boolean myYieldOnNull = true;

  public DerivedAttributeLoader<T> build() {
    return new BuiltBiDerivedLoader<>(
      notNull(myAttributeSpec, "attributeSpec"),
      notNull(myXDependencySpec, "xDependency"),
      notNull(myYDependencySpec, "yDependency"),
      myYieldOnNull,
      notNull(myValueFunction, "derivationFunction"),
      myXTrailFunction,
      myYTrailFunction,
      nullableCollectionOfNonNulls(buildContextDependencies(), "contextDependencies"),
      myCachingStrategy,
      myGlobalTrail);
  }
  
  public BiDerivedAttributeLoaderBuilder<T, X, Y> xDependency(AttributeSpec<X> xDependencySpec) {
    myXDependencySpec = xDependencySpec;
    return this;
  }

  public BiDerivedAttributeLoaderBuilder<T, X, Y> yDependency(AttributeSpec<Y> yDependencySpec) {
    myYDependencySpec = yDependencySpec;
    return this;
  }

  public BiDerivedAttributeLoaderBuilder<T, X, Y> dependencies(AttributeSpec<X> xDependencySpec, AttributeSpec<Y> yDependencySpec) {
    myXDependencySpec = xDependencySpec;
    myYDependencySpec = yDependencySpec;
    return this;
  }

  public BiDerivedAttributeLoaderBuilder<T, X, Y> valueFunction(BiFunction<X, Y, T> derivationFunction) {
    myValueFunction = derivationFunction;
    return this;
  }

  public BiDerivedAttributeLoaderBuilder<T, X, Y> xTrail(Function<X, ItemIdentity> xTrailFunction) {
    myXTrailFunction = xTrailFunction;
    if (xTrailFunction != null) contextDependency(AttributeContextDependency.TRAIL);
    return this;
  }

  public BiDerivedAttributeLoaderBuilder<T, X, Y> yTrail(Function<Y, ItemIdentity> yTrailFunction) {
    myYTrailFunction = yTrailFunction;
    if (yTrailFunction != null) contextDependency(AttributeContextDependency.TRAIL);
    return this;
  }

  public BiDerivedAttributeLoaderBuilder<T, X, Y> yieldOnNull(boolean yieldOnNull) {
    myYieldOnNull = yieldOnNull;
    return this;
  }


  public static class BuiltBiDerivedLoader<T, X, Y> extends AbstractDerivedAttributeLoader<T> {
    private final AttributeSpec<? extends X> myXDependency;
    private final AttributeSpec<? extends Y> myYDependency;
    private final Set<AttributeSpec<?>> myDependencies;
    private final BiFunction<X, Y, T> myDerivationFunction;
    private final Function<X, ItemIdentity> myXTrailFunction;
    private final Function<Y, ItemIdentity> myYTrailFunction;
    private final Set<AttributeContextDependency> myContextDependencies;
    private final AttributeCachingStrategy myCachingStrategy;
    private final boolean myYield;
    private final TrailItemSet myGlobalTrail;

    public BuiltBiDerivedLoader(AttributeSpec<T> spec, AttributeSpec<? extends X> xDependency, AttributeSpec<? extends Y> yDependency,
      boolean yield, BiFunction<X, Y, T> derivationFunction, Function<X, ItemIdentity> xTrailFunction, Function<Y, ItemIdentity> yTrailFunction,
      Set<AttributeContextDependency> contextDependencies, AttributeCachingStrategy cachingStrategy, TrailItemSet globalTrail)
    {
      super(spec);
      myXDependency = xDependency;
      myYDependency = yDependency;
      myYield = yield;
      myDependencies = ImmutableSet.of(myXDependency, myYDependency);
      myDerivationFunction = derivationFunction;
      myXTrailFunction = xTrailFunction;
      myYTrailFunction = yTrailFunction;
      myContextDependencies = contextDependencies;
      myCachingStrategy = cachingStrategy;
      myGlobalTrail = globalTrail;
    }

    @Nullable
    @Override
    public Set<AttributeSpec<?>> getAttributeDependencies() {
      return myDependencies;
    }

    @Nullable
    @Override
    public AttributeValue<T> loadValue(@NotNull DerivedAttributeContext context) {
      AttributeValue<? extends X> xValue = context.getDependencyAttributeValue(myXDependency);
      AttributeValue<? extends Y> yValue = context.getDependencyAttributeValue(myYDependency);
      X x = xValue.getValue();
      Y y = yValue.getValue();
      AttributeValue<T> result;
      if (x == null || y == null) {
        result = myYield ? null : AttributeValue.undefined();
      } else {
        T value = getValue(x, y, context);
        result = value == null && myYield ? null : AttributeValue.ofNullable(value);
      }
      return result;
    }

    @Nullable
    protected T getValue(@NotNull X x, @NotNull Y y, DerivedAttributeContext context) {
      addTrail(x, myXTrailFunction, context);
      addTrail(y, myYTrailFunction, context);
      return myDerivationFunction.apply(x, y);
    }

    private <V> void addTrail(@NotNull V value, Function<V, ItemIdentity> trailFunction, DerivedAttributeContext context) {
      if (trailFunction != null) {
        ItemIdentity trail = trailFunction.apply(value);
        if (trail != null) {
          context.addTrail(trail);
        }
      }
    }

    @Nullable
    @Override
    public Set<AttributeContextDependency> getContextDependencies() {
      return myContextDependencies == null ? super.getContextDependencies() : myContextDependencies;
    }

    @Nullable
    @Override
    public AttributeCachingStrategy getCachingStrategy() {
      return myCachingStrategy == null ? super.getCachingStrategy() : myCachingStrategy;
    }

    @Nullable
    @Override
    public TrailItemSet getGlobalTrail() {
      return myGlobalTrail == null ? super.getGlobalTrail() : myGlobalTrail;
    }
  }
}
