(function(window) {
  var GROUP_KEY = 'com.almworks.jira.structure.sbcolumn';
  var COLUMN_KEY = 'com.almworks.jira.structure.sbcolumn'; // Export renderer providers depend on this.
  var STATUS_BAR_ATTRIBUTE = { id: 'com.almworks.statusbar', format: 'json' }; // Served by StatusBarAttributeProvider.
  var STATUS_ATTRIBUTE = { id: 'status', format: 'id' }; // Served by Structure.

  // Orange, butter, scarlet red, chameleon, plum, sky blue, chocolate, aluminium.
  // http://tango.freedesktop.org/Tango_Icon_Theme_Guidelines
  var PALETTE = [
    '#fcaf3e', '#fce94f', '#ef2929', '#8ae234', '#ad7fa8', '#729fcf', '#e9b96e', '#d3d7cf',
    '#f57900', '#edd400', '#cc0000', '#73d216', '#75507b', '#3465a4', '#c17d11', '#888a85',
    '#ce5c00', '#c4a000', '#a40000', '#4e9a06', '#5c3566', '#204a87', '#8f5902', '#555753'
  ];

  var api = window.almworks.structure.api;
  var $ = window.almworks.structure.$;

  var baseUrl;
  try {
    baseUrl = window.require('wrm/context-path')();
  } catch (e) {
    baseUrl = window['contextPath'] || '';
  }
  if (typeof baseUrl !== 'string') {
    baseUrl = '';
  } else if (baseUrl.length > 0 && baseUrl.substring(baseUrl.length - 1) === '/') {
    baseUrl = baseUrl.substring(0, baseUrl.length - 1);
  }

  /**
   * StatusBarType creates column presets, column instances, and configurators.
   */
  var StatusBarType = api.subClass('StatusBarType', api.ColumnType, {
    getMetadataRequests: function() {
      // We need to know the statuses to create a preset and map status IDs to names.
      // We load them from JIRA's REST API.
      return {
        status: {
          url: baseUrl + '/rest/api/2/status',
          cacheable: true,
          extract: function(response) {
            var result = { order: [], names: {} };
            if ($.isArray(response)) {
              response.forEach(function(status) {
                result.order.push(status.id);
                result.names[status.id] = status.name;
              });
            }
            return result;
          }
        }
      };
    },

    createSwitchTypePreset: function(context) {
      if (isAvailable(context)) {
        // We pre-populate the preset with all statuses and default colors.
        // Thus the user will have a sensible default when switching to our column type.
        var order = context.getMetadata('status').order;
        var statuses = [], colors = [], i;
        for (i = 0; i < order.length; i++) {
          statuses.push(order[i]);
          colors.push(PALETTE[i % PALETTE.length]);
        }
        return {
          key: COLUMN_KEY,
          params: {
            statuses: statuses,
            colors: colors,
            includeItself: true
          }
        };
      }
      return null;
    },

    createAddColumnPresets: function(context) {
      if (isAvailable(context)) {
        // The switch preset is fully configured, so it's OK to use it.
        return [this.createSwitchTypePreset(context)];
      }
      return [];
    },

    createConfigurator: function(context, spec) {
      return isAvailable(context) && isSpecValid(spec) ? new StatusBarConfigurator({context: context, spec: spec}) : null;
    },

    createColumn: function(context, spec) {
      return isAvailable(context) && isSpecValid(spec) ? new StatusBarColumn({context: context, spec: spec}) : null;
    }
  });

  /**
   * @param {!Object} context
   * @returns {boolean}
   */
  function isAvailable(context) {
    return !!(context.getMetadata('status'));
  }

  /**
   * @param {!Object} spec
   * @returns {boolean}
   */
  function isSpecValid(spec) {
    return !!(spec && spec.key === COLUMN_KEY && spec.params
      && $.isArray(spec.params.statuses) && $.isArray(spec.params.colors));
  }

  /**
   * StatusBarColumn renders status bars in the Structure widget.
   */
  var StatusBarColumn = api.subClass('StatusBarColumn', api.Column, {
    getDefaultName: function() {
      return AJS.I18n.getText("sbcolumn.name");
    },
    canShrinkWhenNoSpace: function() {
      return true;
    },

    collectRequiredAttributes: function(attributeSet) {
      attributeSet.requireAttribute(STATUS_BAR_ATTRIBUTE);
      if (!this.spec.params.includeItself) {
        attributeSet.requireAttribute(STATUS_ATTRIBUTE);
      }
    },

    getSortAttribute: function() {
      return { id: 'status', format: 'order' };
    },

    // We could override getCellValueHtml() for a simpler layout, as most columns do.
    // However, we want to look like Structure's Progress column, so we will mimic its layout.
    getCellViewHtml: getCellViewHtml
  });

  var EMPTY_HTML = '<div class="celldiv"></div>';

  function getCellViewHtml(rp) {
    var data = rp.getAttributeValue(STATUS_BAR_ATTRIBUTE);
    if (!data) {
      // Attribute provider disabled, network failure, showing aggregates for a historic version, etc.
      // Columns should not fail when there's no data.
      return EMPTY_HTML;
    }

    if (!this.spec.params.includeItself) {
      var status = rp.getAttributeValue(STATUS_ATTRIBUTE);
      if (status && data[status] > 0) {
        data = $.extend({}, data);
        data[status] -= 1;
      }
    }

    // Must be present, otherwise StatusBarType shouldn't have created the column.
    var meta = this.context.getMetadata('status');

    var statuses = this.spec.params.statuses || meta.order;
    var percent = distribute(data, statuses, 100);
    if (!percent || $.isEmptyObject(percent)) {
      // None of the selected statuses is in the data.
      return EMPTY_HTML;
    }

    // Generating HTML.
    var colors = this.spec.params.colors || PALETTE;
    var html = '<div class="celldiv"><div class="simpleProgress">';
    var i, s, p;
    for (i = 0; i < statuses.length; i++) {
      s = statuses[i];
      p = percent[s];
      if (p > 0) {
        html += AJS.template('<div style="width: {percent}%; background-color: {color};" title="{name}: {count}"></div>')
          .fill({
            percent: p,
            color: colors[i % colors.length],
            name: meta.names[s] || s,
            count: data[s] || '0'
          }).toString();
      }
    }
    html += '</div></div>';
    return html;
  }

  /**
   * @param {!Object.<string, number>} values A map from strings to numbers.
   * @param {!Array.<string>} keys Selected keys for the <code>values</code> map. Other keys in the map are ignored.
   * @param {number} amount The total amount to distribute.
   * @returns {!Object.<string, number>} A new map from strings to numbers. Its keys are those present in both
   * <code>values</code> and <code>keys</code>. Its values are proportional to the numbers in the input map and sum up
   * to <code>amount</code>.
   */
  function distribute(values, keys, amount) {
    var i, value, total = 0, max = 0;
    for (i = 0; i < keys.length; i++) {
      value = values[keys[i]];
      if (value > 0) {
        total += value;
        max = Math.max(max, value);
      }
    }

    if (total === 0) {
      return {};
    }

    var result = {}, remaining = amount, p;
    for (i = 0; i < keys.length; i++) {
      value = values[keys[i]];
      if (value > 0 && value < max) {
        p = Math.round(value / total * amount);
        result[keys[i]] = p;
        remaining -= p;
      }
    }

    var last;
    for (i = 0; i < keys.length; i++) {
      value = values[keys[i]];
      if (value === max) {
        last = keys[i];
        p = Math.min(Math.round(value / total * amount), remaining);
        result[last] = p;
        remaining -= p;
      }
    }

    if (remaining > 0) {
      result[last] += remaining;
    }

    return result;
  }

  /**
   * StatusBarConfigurator is responsible for column configuration.
   * The bulk of the code is in option classes below.
   */
  var StatusBarConfigurator = api.subClass('StatusBarConfigurator', api.ColumnConfigurator, {
    init: function() {
      this.spec.key = COLUMN_KEY;
      this.spec.params || (this.spec.params = {});
    },
    getColumnTypeName: function() {
      return AJS.I18n.getText("sbcolumn.name");
    },
    getGroupKey: function() {
      return GROUP_KEY;
    },
    getOptions: function() {
      return [new StatusesOption({ configurator: this }), new IncludeItselfOption({ configurator: this })];
    }
  });

  /**
   * StatusesOption provides the UI to select the statuses for the column
   * and set their order and colors.
   */
  var StatusesOption = api.subClass('StatusesOption', api.ColumnOption, {
    title: AJS.I18n.getText("sbcolumn.statuses"),

    init: function() {
      this.div$ = null
    },

    createInput: function(div$) {
      var params = this.spec.params;
      var statuses = $.merge([], params.statuses);
      var meta = this.context.getMetadata('status');
      var i, s;

      // Adding unselected statuses at the end.
      for (i = 0; s = meta.order[i]; i++) {
        if ($.inArray(s, statuses) < 0) {
          statuses.push(s);
        }
      }

      // Generating HTML.
      var colorHtml = '<input type="text" size="7" maxlength="7" pattern="#[0-9a-fA-F]{6}" class="sbi-color">';
      if ($.browser.chrome) {
        colorHtml = '<input type="color" class="sbi-color">';
      }

      var html = '<div class="checkbox">';
      for (i = 0; s = statuses[i]; i++) {
        html += AJS.template(
          '<div class="st-col-drag" data-status="{status}">' +
            '<div class="st-col-drag-handle" title="{dragHint}"><div class="st-col-drag-icon"></div></div>' +
            '<input type="checkbox" id="sbi-check-{status}" class="sbi-check">&nbsp;' +
            '{color}&nbsp;' +
            '<label for="sbi-check-{status}">{name}</label>' +
            '</div>')
          .fill({
            status: s,
            name: meta.names[s] || s,
            'color:html': colorHtml,
            dragHint: AJS.I18n.getText("sbcolumn.drag-hint")
          }).toString();
      }
      html += '</div>';

      // Adding to div$ and attaching listeners.
      $(html).appendTo(div$).on('change', 'input', update).sortable({ axis: 'y', update: update });
      this.div$ = div$;

      // Fill spec params from input values.
      function update() {
        params.statuses = [];
        params.colors = [];
        div$.find('input.sbi-check:checked').each(function() {
          var parent$ = $(this).closest('div[data-status]');
          params.statuses.push(parent$.attr('data-status'));
          params.colors.push(parent$.find('input.sbi-color').val());
        });
        div$.trigger('notify');
      }
    },

    notify: function() {
      var statuses = this.spec.params.statuses;
      var colors = this.spec.params.colors;
      var meta = this.context.getMetadata('status');

      // Set input values from spec params.
      this.div$.find('div[data-status]').each(function() {
        var this$ = $(this);
        var s = this$.attr('data-status');
        var i = $.inArray(s, statuses);
        if (i >= 0) {
          this$.find('input.sbi-check').prop('checked', true);
          this$.find('input.sbi-color').val(colors[i] || defaultColor(s));
        } else {
          this$.find('input.sbi-check').prop('checked', false);
          this$.find('input.sbi-color').val(defaultColor(s));
        }
      });

      return true; // Show this option.

      function defaultColor(s) {
        var i = $.inArray(s, meta.order);
        return PALETTE[i % PALETTE.length];
      }
    },

    isInputValid: function() {
      var p = this.spec.params;
      return !!(p && $.isArray(p.statuses) && p.statuses.length > 0
        && $.isArray(p.colors) && p.colors.length === p.statuses.length);
    }
  });

  /**
   * IncludeItselfOption provides the UI for the "Include itself" checkbox
   */
  var IncludeItselfOption = api.subClass('IncludeItselfOption', api.ColumnOption, {
    init: function() {
      this.checkbox$ = null;
    },

    createInput: function(div$) {
      this.checkbox$ = div$.append(
          AJS.template('<div class="checkbox"><label><input type="checkbox">&nbsp;{label}</label></div>')
            .fill({ label: AJS.I18n.getText("sbcolumn.include-itself") })
            .toString()).find('input');

      var params = this.spec.params;
      this.checkbox$.on('change', function() {
        if ($(this).is(':checked')) {
          params.includeItself = true;
        } else {
          delete params.includeItself;
        }
        div$.trigger('notify');
      });
    },

    notify: function() {
      this.checkbox$.prop('checked', !!this.spec.params.includeItself);
      return true;
    }
  });

  // Finally, we let Structure know about our column group and column type.
  api.registerColumnGroup({ groupKey: GROUP_KEY, title: AJS.I18n.getText("sbcolumn.name"), order: 1000 });
  api.registerColumnType(new StatusBarType(), COLUMN_KEY);
})(window);
